# -*- coding: utf-8 -*-
import urllib
import uuid

import requests
from lxml import etree
from django.template import loader

from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every, BadPartnerResponse, PartnerErrorTypes
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import parse_datetime_without_seconds

KLASS_MAP = {
    'economy': 'economy',
    'business': 'business',
}

NEMO_ERRORS = {
    '0': PartnerErrorTypes.ERROR,
    '1': PartnerErrorTypes.SYSTEM_ERROR,
    '2': PartnerErrorTypes.ACCESS_DENIED,
    '4': PartnerErrorTypes.AUTHORIZATION_ERROR,
    '1000': PartnerErrorTypes.IVALID_REQUEST_STRUCTURE,
    '1002': PartnerErrorTypes.SYSTEM_ERROR,
    '1003': PartnerErrorTypes.SABRE_GDS_ERROR,
    '1007': PartnerErrorTypes.GDS_CONNECTION_ERROR,
    '1014': PartnerErrorTypes.AMADEUS_GDS_ERROR,
    '1015': PartnerErrorTypes.SIRENA_GDS_ERROR,
    '1016': PartnerErrorTypes.GALILEO_GDS_ERROR,
    '1018': PartnerErrorTypes.SITA_GABRIEL_GDS_ERROR,
    '1019': PartnerErrorTypes.VIP_SERVICE_GDS_ERROR,
    '1020': PartnerErrorTypes.NI_GDS_ERROR,
    '1021': PartnerErrorTypes.PEGASUS_GDS_ERROR,
    '1022': PartnerErrorTypes.TF_GDS_ERROR,
    '1026': PartnerErrorTypes.SEARCHES_LIMIT_ERROR,
    '1027': PartnerErrorTypes.SEARCHES_PER_TICKET_LIMIT_ERROR,
    '1029': PartnerErrorTypes.TARIFF_RULES_PARSING_ERROR,
    '1030': PartnerErrorTypes.NETWORK_ERROR,
    '1031': PartnerErrorTypes.GALILEO_GDS_ERROR,
    '1032': PartnerErrorTypes.USER_CONFIGURATION_ERROR,
}


class NemoImporter(object):
    def __init__(self, client_id, api_key, search_url, utm_source, logger, action=u'IAgencyAPI/search'):
        self._client_id = client_id
        self._api_key = api_key
        self._search_url = search_url
        self._utm_source = utm_source
        self._action = action
        self._logger = logger

    def query(self, tracker, q, unquote_target_url=True):
        # TODO(u-jeen): make unquote_target_url=False the default value or stop unquoting altogether
        response = self._get_data(tracker, q)

        return self._parse(
            response,
            response.content,
            flight_fabric=q.importer.flight_fabric,
            query=q,
            unquote_target_url=unquote_target_url,
        )

    def _build_xml(self, xml_template_file, params):
        query_xml = loader.render_to_string(xml_template_file, params)

        return ''.join(filter(None, [
            line.strip() for line in query_xml.splitlines()
        ]))

    def _get_data(self, tracker, query):
        query_xml = self._build_xml(
            'partners/nemo2.xml',
            self._build_params(query)
        )

        self._logger.info('Send request: %r', query_xml)

        r = tracker.wrap_request(
            requests.post,
            self._search_url,
            headers={
                'SOAPAction': 'urn:{}'.format(self._action),
                'Content-Type': 'text/xml; charset=utf-8',
            },
            data=query_xml.encode('utf-8'),
            verify=False
        )

        return r

    def _build_params(self, q):
        return {
            'forward_date': q.date_forward.strftime('%Y-%m-%d'),
            'return_date': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or '',
            'od_type': q.date_backward and 'RT' or 'OW',
            'iata_from': q.iata_from,
            'iata_to': q.iata_to,

            'travellers': dict(filter(lambda (k, v): v, {
                'ADT': q.passengers.get('adults', 0),
                'CNN': q.passengers.get('children', 0),
                'INF': q.passengers.get('infants', 0),
            }.items())),

            'klass': KLASS_MAP[q.klass],

            'client_id': self._client_id,
            'api_key': self._api_key,
            'session_id': uuid.uuid4(),
            'action': self._action,
            'search_url': self._search_url
        }

    def _parse(self, response, xml, flight_fabric, query, unquote_target_url=True):
        tree = etree.fromstring(xml)
        self._check_errors(tree, response)
        variants = []

        for e in sleep_every(tree.xpath('//Response/SearchFlights/Flights/Flight')):
            v = Variant()

            forward_segments = []
            backward_segments = []

            segments = e.xpath('Segments/Segment')

            for s in segments:
                if s.get('SegGroupNum', '0') == '0':
                    forward_segments.append(s)
                else:
                    backward_segments.append(s)

            tariffs = self._parse_tariff(e)
            v.forward.segments = self._parse_segments(
                flight_fabric, forward_segments, query, tariffs
            )
            v.backward.segments = self._parse_segments(
                flight_fabric, backward_segments, query, tariffs
            )

            v.tariff = Price(float(e.findtext('TotalPrice')))

            v.klass = query.klass

            v.charter = any(f.charter for f in v.all_segments)

            raw_url = e.findtext('URL')
            booking_info = {'FlightId': e.get('FlightId')}

            url = self._generate_url(raw_url, unquote_target_url)
            v.url = url
            v.order_data = {
                'url': url,
                'booking_info': booking_info,
            }

            variants.append(v)

        return variants

    def _check_errors(self, tree, response):
        # type: (etree.Element, str) -> None
        errors = set()
        error_elements = tree.xpath('//Errors/Error/Code')
        if not error_elements:
            return

        for error_code in error_elements:
            if error_code.text not in NEMO_ERRORS:
                self._logger.warning('Unknown error code %s', error_code.text)
            else:
                errors.add(NEMO_ERRORS[error_code.text])

        raise BadPartnerResponse('nemo', response, ','.join(errors))

    def _generate_url(self, raw_url, unquote_target_url=True):
        # type: (str) -> str
        url = urllib.unquote(raw_url) if unquote_target_url else raw_url
        if self._utm_source:
            url = url_complement_missing(
                url,
                {'utm_source': self._utm_source},
            )

        return url

    def _parse_tariff(self, etree_flight):
        # type: (etree.Element) -> dict
        tariffs = {}
        for tariff in etree_flight.xpath('PricingInfo/PassengerFare/Tariffs/Tariff'):
            seg_num = tariff.get('SegNum')
            if seg_num:
                tariffs[seg_num] = tariff.get('Code')
        return tariffs

    def _parse_segments(self, flight_fabric, items, q, tariffs):
        flights = []

        for item in items:
            flights.append(flight_fabric.create(
                station_from_iata=item.findtext('DepAirp'),
                station_to_iata=item.findtext('ArrAirp'),
                local_departure=self._extract_datetime(item, 'DepDateTime'),
                local_arrival=self._extract_datetime(item, 'ArrDateTime'),
                company_iata=item.findtext('MarkAirline'),
                pure_number=item.findtext('FlightNumber'),
                charter=item.findtext('isCharter') == 'true',
                baggage=self._get_baggage(item, q),
                fare_code=tariffs.get(item.get('SegNum'))
            ))

        return flights

    def _get_baggage(self, segment, q):
        """
        <Segment>
         ...
         <BaggageAllowances>
          <BaggageAllowance>
           <PassengerType>ADT</PassengerType>
           <Value>0</Value>
           <Measurement>pc</Measurement>
          </BaggageAllowance>
         </BaggageAllowances>
         ...
        </Segment>
        """
        try:
            value = segment.findtext('BaggageAllowances/BaggageAllowance/Value')
            if value is None:
                return Baggage.from_partner()
            measurement = segment.findtext(
                'BaggageAllowances/BaggageAllowance/Measurement')
            if measurement.lower() == 'kg':
                return Baggage.from_partner(weight=value)
            if measurement.lower() == 'pc':
                return Baggage.from_partner(pieces=value)
        except Exception:
            self._logger.warning(
                'Baggage parsing fail qkey: [{}] [{}]'.format(q.qkey, segment)
            )
        return Baggage.from_partner()

    def _extract_datetime(self, xml_node, tag):
        raw_date = xml_node.findtext(tag)
        return parse_datetime_without_seconds(raw_date)
