# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import datetime, timedelta
from urllib import urlencode

import requests
import ujson as json
from django.conf import settings

from travel.avia.ticket_daemon.ticket_daemon.api.flights import min_class, Variant
from travel.avia.ticket_daemon.ticket_daemon.api.selfbook.partner_national_directions import (
    SelfBookPartnerNationalDirectionRules)
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse, sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage, BaggageParser


OTT_URL = 'http://api.twiket.com/searching/startSync4a/'
LOGIN = 'api@yandex'
PASSWORD = partner_secret_storage.get(
    importer_name='onetwotrip32', namespace='PASSWORD'
)
NO_BAGGAGE = frozenset({'0NO', 'NO'})

log = logging.getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

"""
In [1]: from travel.avia.library.python.common.models.partner import *
In [2]: p = Partner.objects.get(code='onetwotrip')
In [3]: p.query_module_name = 'onetwotrip3'
In [4]: p.save()
"""


@QueryTracker.init_query
def query(tracker, q):
    data = get_data(tracker, q)
    variants = parse_response(data, q)
    return variants


def get_data(tracker, q):
    """
    * route — is a route customer going to travel, consists of directions.
        Direction structure is [DDMM][FROM][TO], where DD is a day of travel, MM is
        a month of travel, FROM/TO are IATA city or airport codes

    * ad, cn, in — is a customer's desired passengers count. AD — adult passengers
        count, CN — child passengers count ,IN — infant passengers count

    * cs — is a desired cabin class for flight. Can be E — economy, B — business, F —
        first
    """

    r = tracker.wrap_request(
        requests.get,
        OTT_URL,
        params=skip_None_values({
            'route': u''.join(filter(None, [
                _format_direction(q.iata_from, q.iata_to, q.date_forward),
                _format_direction(q.iata_to, q.iata_from, q.date_backward) if q.date_backward else None,
            ])),
            'ad': q.passengers.get('adults', 0),
            'cn': q.passengers.get('children', 0),
            'in': q.passengers.get('infants', 0),
            'cs': KLASS_MAP.get(q.klass, 'E'),
            'showDeeplink': 'true',
            'source': ('yandex' if q.national_version == 'ru' else
                       'yandex_{}'.format(q.national_version)),
            'searchSource': 'meta',
        }),
        auth=requests.auth.HTTPBasicAuth(LOGIN, PASSWORD)
    )
    data = json.loads(r.content)
    if 'trps' not in data:
        raise BadPartnerResponse('onetwotrip31 no "trps" key', r)
    return data


def _format_direction(iata_from, iata_to, dt):
    return u'{}{}{}'.format(dt.strftime('%d%m'), iata_from, iata_to)


KLASS_MAP = {'economy': 'E', 'business': 'B'}
KLASS_REVERSE_MAP = {
    'E': 'economy',
    'W': 'economy',
    'B': 'business',
    'F': 'business',
}


@pipe(list)
def parse_response(data, q):
    trips = data['trps']
    raw_flights = [parse_trip(q, trip) for trip in trips]

    selfbook_ruler = SelfBookPartnerNationalDirectionRules(
        # Пока только один партнёр в этом импортёре, можно доставать первый
        q.importer.partners[0],
        q.national_version, q.iata_from, q.iata_to
    )

    adults = q.passengers.get('adults', 0)
    # youngsters_ages: - separted list of ages of kids, ie 0-2-6 (3 kids of age
    # 0, 2 and 6)
    youngsters_ages = '-'.join(map(str, (
        [6] * q.passengers.get('children', 0) +
        [1] * q.passengers.get('infants', 0)
    )))

    for fare in sleep_every(data['frs']):
        v = Variant()
        v.tariff = Price(float(fare['prcInf']['amt']), fare['prcInf']['cur'])

        v.forward.segments = [
            cook_flight(q, raw_flights[fare_trip['id']], fare_trip)
            for fare_trip in fare['dirs'][0]['trps']
        ]
        # Пропускаем варианты в которых некоторые рейсы не распарсились
        if not all(v.forward.segments):
            continue
        if q.date_backward:
            v.backward.segments = [
                cook_flight(q, raw_flights[fare_trip['id']], fare_trip)
                for fare_trip in fare['dirs'][1]['trps']
            ]
            if not all(v.backward.segments):
                continue

        v.klass = min_class(v.get_klasses())

        v.order_data = {}
        if selfbook_ruler.is_selfbook_applicable(v):
            v.order_data['sb'] = json.dumps(selfbook_deeplink_params(
                trips, fare, v.tariff,
                v.forward.segments[0].station_from_iata,
                v.forward.segments[-1].station_to_iata,
                adults, youngsters_ages
            ))
        else:
            v.order_data['url'] = fare['deeplink']

        yield v


def selfbook_deeplink_params(trips, fare, tariff, iata_from, iata_to, adults, youngsters_ages):
    forward, backward = ([
        tuple(trips[segment['id']] for segment in direction['trps'])
        for direction in fare['dirs']
    ] + [()])[:2]
    return dict(filter(None, [
        ('version', '3'),
        ('fares', '{price:.2f}-{currency}-{vendor_id}'.format(
            price=tariff.value,
            currency=tariff.currency.replace('RUR', 'RUB'),
            vendor_id='1262',  # 1262 for ott russia
        )),
        ('n_adults', str(adults)),
        ('youngsters_ages', youngsters_ages),
        ('combo', '-'.join(['0'] * sum(map(len, [forward, backward])))),  # 0
        ('out', '---'.join(format_leg(f) for f in forward)),  # leg1---leg2---leg3
        ('home', '---'.join(format_leg(f) for f in backward)) if backward else None,
        #            ^ leg4---leg5---leg6
        ('curr', fare['prcInf']['cur']),
        ('route', '{}{}'.format(iata_from, iata_to)),  # [airport_list]

        # ('res', residence),  # RU
        # ('lang', lang),  # ru
        # ('utm_source', 'avia.yandex.ru'),  # Dohop.ru
    ]))


# def selfbook_deeplink(trips, fare, tariff, iata_from, iata_to):
#     params = selfbook_deeplink_params(trips, fare, tariff, iata_from, iata_to)
#     return 'https://avia.yandex.ru/booking/ticket/?' + urlencode(params)


def format_leg(trip):
    return (
        '{airport_code_out}{airport_code_in}'
        '{date_out}{departure_time}{day_delta}'
        '{arrival_time}{duration}{flight}').format(
            airport_code_out=trip['from'],
            airport_code_in=trip['to'],
            date_out=trip['stDt'],
            departure_time=trip['stTm'],
            day_delta=trip.get('dayChg', '0'),
            arrival_time=trip['endTm'],
            duration=trip['fltTm'],
            flight='{}{}'.format(trip['airCmp'], trip['fltNm']),
        )


DT_FMT = '%Y%m%d%H%M'


def parse_day_time(trip, day_key, tm_key):
    day = trip[day_key]
    if not day:
        raise ValueError('No day part %r' % day_key)

    tm = trip[tm_key]
    if not tm:
        raise ValueError('No time part %r' % tm_key)

    return datetime.strptime(day + tm, DT_FMT)


def cook_flight(q, raw_flight, fare_trip):
    srv_cls = fare_trip['srvCls']
    if srv_cls not in KLASS_REVERSE_MAP:
        log.warning('Unknown service class %r', srv_cls)
        return None

    return q.importer.flight_fabric.create(
        klass=KLASS_REVERSE_MAP[srv_cls],
        baggage=parse_baggage(fare_trip.get('bg')),
        **raw_flight._asdict()
    )


RawFlight = namedtuple('RawFlight', [
    'station_from_iata', 'station_to_iata',
    'local_departure', 'local_arrival',
    'company_iata', 'pure_number',
])


def parse_baggage(baggage):
    # type: (str) -> Baggage
    if not baggage:
        return Baggage.from_partner()
    if baggage in NO_BAGGAGE:
        return Baggage.from_partner(included=False)
    return baggage_parser.parse_from_string(baggage)


def parse_trip(q, trip):
    try:
        local_departure = parse_day_time(trip, 'stDt', 'stTm')
        local_arrival = (parse_day_time(trip, 'stDt', 'endTm') +
                         timedelta(days=int(trip.get('dayChg', 0))))

    except (KeyError, ValueError) as e:
        log.warning('Parse datetimes error: %r %s %r', e, q.id, trip)
        return None

    return RawFlight(
        station_from_iata=trip['from'].encode('utf-8'),
        station_to_iata=trip['to'].encode('utf-8'),
        local_departure=local_departure,
        local_arrival=local_arrival,
        company_iata=trip['airCmp'].encode('utf-8'),
        pure_number=trip['fltNm'],
    )


RESIDENCIES_MAP = {'ru': 'RU', 'ua': 'UA', 'tr': 'TR', 'com': 'GB'}


def book(order_data):
    if 'url' in order_data:
        return order_data['url']

    # Selfbook params
    if 'sb' in order_data:
        params = json.loads(order_data['sb'])

        lang = order_data['lang']
        national_version = order_data['national_version']
        avia_hosts = settings.AVIA_HOST_BY_NATIONAL_VERSION
        avia_host = avia_hosts.get(national_version, avia_hosts['ru'])

        params.update({
            'res': RESIDENCIES_MAP.get(national_version, RESIDENCIES_MAP['ru']),
            'lang': lang,
            'utm_source': avia_host,
        })
        return 'https://avia.yandex.ru/booking/ticket/?' + urlencode(params)
