# -*- coding: utf-8 -*-
import logging
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, OperatingFlight
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser

OTT_URL = 'https://api.twiket.com/redirectreseller/getsearchresult/'
LOGIN = 'api@yandex'
PASSWORD = partner_secret_storage.get(
    importer_name='onetwotrip33', namespace='PASSWORD'
)

log = logging.getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

KLASS_MAP = {'economy': 0, 'business': 1}
DT_FMT = '%Y-%m-%dT%H:%M:%S'


@QueryTracker.init_query
def query(tracker, q):
    tree = get_data(tracker, q)
    variants = parse_response(tree, q)

    return variants


def build_search_params(q):
    route_builder = [q.date_forward.strftime('%d%m'), q.iata_from.encode('utf-8'), q.iata_to.encode('utf-8')]
    if q.date_backward:
        route_builder.append(q.date_backward.strftime('%d%m'))
    route = "".join(route_builder)
    return skip_None_values({
        'route': route,
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'trip_class': KLASS_MAP[q.klass],
        'range': 'true',
        'showBaggage': 'true',
        'identifier': 'meta',
    })


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        OTT_URL,
        params=build_search_params(q),
        auth=requests.auth.HTTPBasicAuth(LOGIN, PASSWORD)
    )

    tree = etree.fromstring(r.content)
    return tree


def parse_response(tree, q):
    variants = []

    for variant in sleep_every(list(tree.xpath('/SearchResult/Offers/Item'))):
        try:
            v = Variant()

            v.forward.segments, v.backward.segments = parse_segments(variant.xpath('./Flights/Item'),
                                                                     q.importer.flight_fabric)

            v.tariff = Price(float(variant.attrib.get('Price')), variant.attrib.get('currency'))
            v.klass = q.klass
            v.url = variant.attrib.get('Url')
            v.order_data = {'url': v.url}
            variants.append(v)
        except (ValueError, KeyError, TypeError) as e:
            log.error('Failed to parse variant: %s Error message: %s',
                      etree.tostring(variant, pretty_print=True),
                      e.message)
    return variants


def parse_segments(items, flight_fabric):
    forward, backward = [], []
    for item in items:
        flight = flight_fabric.create(
            station_from_iata=item.attrib.get('Origin'),
            station_to_iata=item.attrib.get('Destination'),
            local_departure=datetime.strptime(item.attrib.get('Depart'), DT_FMT),
            local_arrival=datetime.strptime(item.attrib.get('Arrive'), DT_FMT),
            company_iata=item.attrib.get('Code'),
            pure_number=item.attrib.get('Num'),
            fare_code=item.attrib.get('Basis'),
            baggage=baggage_parser.parse_from_string(item.attrib.get('Baggage')),
            operating=OperatingFlight(company_iata=item.attrib.get('OPCode')),
        )
        if item.attrib.get('IsDeparture') == 'true':
            forward.append(flight)
        else:
            backward.append(flight)
    return forward, backward
