# coding: utf8
import itertools
from collections import namedtuple
from datetime import datetime
from logging import getLogger
from typing import Iterable, List

import requests
from django.utils.http import urlencode

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage

CLASS_MAP = {
    'economy': 'ECONOMY',
    'business': 'BUSINESS',
    'first': 'FIRST',
}

OZON_API_URL = 'http://partners.ozon.travel/search_v1_0/flight/'
OZON_DEEPLINK_URL = 'http://www.ozon.travel/partnertools/deeplink_v1_0/flight/'
OZON_PARTNER = 'yandex'
OZON_UTM_MEDIUM = 'metasearch'
OZON_UTM_SOURCES = {
    'default': 'avia.yandex.ru',
    'kz': 'avia.yandex.kz',
}
VariantInfo = namedtuple('VariantInfo', (
    'segments', 'class_codes', 'marketing_flight_numbers', 'fare_basis',
))

log = getLogger(__name__)


@QueryTracker.init_query
def query(tracker, q):
    # type: (QueryTracker, Query) -> List[Variant]
    response = get_data(tracker, q)

    variants = list(parse_response(response, q))

    return variants


def get_data(tracker, q):
    # type: (QueryTracker, Query) -> dict
    r = tracker.wrap_request(
        requests.get,
        OZON_API_URL,
        params=build_query_params(q),
    )

    # Remove BOM
    r.encoding = 'utf-8-sig'

    return r.json()


def build_query_params(q):
    # type: (Query) -> dict
    params = {}
    # 1. собираем Flight
    if q.date_backward:
        params['Flight'] = '%s%s%s' % (
            q.iata_from.encode('utf-8'),
            q.iata_to.encode('utf-8'),
            q.iata_from.encode('utf-8')
        )

        params['Date2'] = q.date_backward.strftime('%Y-%m-%d')

    else:
        params['Flight'] = '%s%s' % (
            q.iata_from.encode('utf-8'),
            q.iata_to.encode('utf-8')
        )

    params['Date1'] = q.date_forward.strftime('%Y-%m-%d')
    params['Dlts'] = q.passengers.get('adults', 0)
    params['Children'] = q.passengers.get('children', 0)
    params['Infants'] = q.passengers.get('infants', 0)
    params['ServiceClass'] = CLASS_MAP[q.klass]

    return params


def parse_response(data, q):
    # type: (dict, Query) -> Iterable[Variant]

    _query = build_query_params(q)
    _query['partner'] = OZON_PARTNER
    _query['utm_medium'] = OZON_UTM_MEDIUM
    _query['utm_source'] = OZON_UTM_SOURCES.get(q.national_version, OZON_UTM_SOURCES['default'])
    if q.national_version == 'kz':
        _query['market'] = 'KZ'

    prefix_url = OZON_DEEPLINK_URL + '?' + urlencode(_query)

    search_id = data.get('searchId')

    for tariff in data['data']:
        klass = tariff['serviceClass'].lower()

        cost = Price(float(tariff['prices'][0]))
        int_cost = int(cost.value)

        self_connect = tariff.get('SelfConnect', False)
        forward_flights = (
            parse_flights(f, q.importer.flight_fabric, self_connect)
            for f in tariff['segments'][0]['flights']
        )
        backward_flights = (
            parse_flights(f, q.importer.flight_fabric, self_connect)
            for f in tariff['segments'][1]['flights']
        ) if q.date_backward else [None]

        for forward, backward in sleep_every(itertools.product(forward_flights, backward_flights)):
            v = Variant()
            v.forward.segments = forward.segments

            if q.date_backward:
                v.backward.segments = backward.segments

            v.tariff = cost
            v.klass = klass
            v.charter = tariff.get('charter', False)

            additional_params = {
                'FlightsNo': forward.marketing_flight_numbers,
                'FlightsClassCode': forward.class_codes,
                'fareBasis': forward.fare_basis,
                'prices': int_cost,
            }

            if search_id:
                additional_params['searchId'] = search_id

            if q.date_backward:
                additional_params['FlightsNo'] += ';' + backward.marketing_flight_numbers
                additional_params['FlightsClassCode'] += ';' + backward.class_codes
                additional_params['fareBasis'] += ';' + backward.fare_basis

            v.order_data = {
                'url': '{}&{}'.format(prefix_url, urlencode(additional_params)),
            }

            yield v


def parse_flights(flight, flight_fabric, self_connect):
    # type: (dict, FlightFabric, bool) -> VariantInfo
    flights = []
    class_codes = []
    marketing_flight_numbers = []
    fare_basis = []

    for leg in flight['flightLegs']:
        marketing_airline_code = leg['airlineCode']
        pure_number = leg['flightNo']

        flights.append(flight_fabric.create(
            company_iata=marketing_airline_code,
            pure_number=pure_number,
            station_from_iata=leg['from']['code'],
            station_to_iata=leg['to']['code'],
            local_departure=extract_datetime(leg['fromDate'], leg['fromTime']),
            local_arrival=extract_datetime(leg['toDate'], leg['toTime']),
            fare_code=leg.get('originalFareBasis'),
            baggage=get_baggage(leg),
            selfconnect=self_connect,
        ))
        class_codes.append(leg.get('classCode'))
        marketing_flight_numbers.append(
            u'{}{}'.format(marketing_airline_code, pure_number),
        )
        fare_basis.append(leg.get('fareBasis'))

    return VariantInfo(
        flights,
        ','.join(class_codes).encode('utf-8'),
        ','.join(marketing_flight_numbers).encode('utf-8'),
        ','.join(fare_basis).encode('utf-8'),
    )


def extract_datetime(date_str, time_str):
    # type: (str, str) -> datetime
    return datetime(
        int(date_str[0:4]), int(date_str[5:7]), int(date_str[8:10]),
        int(time_str[0:2]), int(time_str[3:5]),
    )


def get_baggage(flight):
    # type: (dict) -> Baggage
    try:
        value = flight['luggage']['value']
        unit = flight['luggage']['unit']
        if unit == 'NO':
            return Baggage.from_partner(included=False)
        elif unit == 'PLACE':
            return Baggage.from_partner(pieces=int(value))
        elif unit == 'KILOGRAM':
            return Baggage.from_partner(weight=int(value))
        else:
            log.info('Unknown baggage unit %r', unit)

    except Exception as e:
        log.info('Baggage parsing exception: [%s] on element %r', e, flight)

    return Baggage.from_partner()
