# -*- coding: utf-8 -*-
import itertools
import json
import logging
import re
from datetime import datetime

import requests

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse, sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = logging.getLogger(__name__)

SEARCH_URL = 'http://api.pilot.ua/searches/offers'

KLASS_MAP = {'economy': 'Economy', 'business': 'Business'}
KLASS_REVERSE_MAP = dict(map(reversed, KLASS_MAP.items()))


def validate_query(q):
    q.validate_passengers(adults=5, children=5, count=6)
    q.validate_klass(KLASS_MAP)

    if not getattr(q, 'iata_real_from', False):
        raise QueryIsNotValid('No iata_real_from')

    if not getattr(q, 'iata_real_to', False):
        raise QueryIsNotValid('No iata_real_to')


@QueryTracker.init_query
def query(tracker, q):
    json_response = get_data(tracker, q)
    variants = parse_response(json_response, q)

    return variants


def get_data(tracker, q):
    params = {
        'search': {
            'adults': q.passengers.get('adults', 0),
            'children': q.passengers.get('children', 0),
            'infants': q.passengers.get('infants', 0),
            'cabin_class': KLASS_MAP[q.klass],
            'legs_attributes': {
                0: {
                    'from': q.iata_from.encode('utf-8'),
                    'leg_id': 0,
                    'to': q.iata_to,
                    'at': q.date_forward.strftime('%Y-%m-%d')
                }
            }
        }
    }

    if q.date_backward:
        params['search']['legs_attributes'].update({
            1: {
                'from': q.iata_to,
                'leg_id': 1,
                'to': q.iata_from,
                'at': q.date_backward.strftime('%Y-%m-%d')
            }
        })

    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        headers={'Content-Type': 'application/json'},
        data=json.dumps(params)
    )

    json_response = r.json()

    if 'search' not in json_response:
        raise BadPartnerResponse('pilotua', r)

    return json_response


def parse_response(json_response, q):
    variants = []

    offers = json_response['search']['offers']
    klass = json_response['search']['cabin_class']

    for offer in sleep_every(offers):
        v = Variant()

        v.forward.segments = parse_segment(filter(
            lambda s: int(s['booking_info']['group']) == 0,
            offer['segments']
        ))

        if q.date_backward:
            v.backward.segments = parse_segment(filter(
                lambda s: int(s['booking_info']['group']) == 1,
                offer['segments']
            ))

        currency = offer['grand_total'][:3]
        price = float(offer['grand_total'][3:])

        v.tariff = Price(price, currency)

        v.klass = KLASS_REVERSE_MAP[klass]

        air_reservation = json.dumps(
            build_air_reservation(offer['segments'], q))

        v.order_data = {
            'air_reservation': air_reservation,
        }

        variants.append(v)

    return variants


DT_FMT = '%Y-%m-%dT%H:%M'
DT_LEN = len(datetime.now().strftime(DT_FMT))


@elementwise
def parse_segment(segment):
    f = IATAFlight()

    f.local_departure = datetime.strptime(
        segment['booking_info']['departure_time'][:DT_LEN], DT_FMT)

    f.local_arrival = datetime.strptime(
        segment['booking_info']['arrival_time'][:DT_LEN], DT_FMT)

    f.company_iata = segment['booking_info']['carrier']
    f.number = f.company_iata + ' ' + segment['booking_info']['flight_number']

    f.station_from_iata = extract_iata(segment['from'])
    f.station_to_iata = extract_iata(segment['to'])

    return f


def extract_iata(text):
    # No sirena codes
    m = re.search(r'\(([A-Z]{3})\)', text)

    return m.group(1) if m else None


RESERVATION_PASSENGERS = [
    ('adults', 'adult_passengers_attributes'),
    ('children', 'child_passengers_attributes'),
    ('infants', 'infant_passengers_attributes'),
]


def build_air_reservation(segments, q):
    air_reservation = {
        'segments_attributes': {
            j: s['booking_info'] for j, s in enumerate(segments)
        }
    }

    index = itertools.count()

    for which_passengers, passengers_attrname in RESERVATION_PASSENGERS:
        passengers_count = int(q.passengers.get(which_passengers, 0))

        if passengers_count:
            air_reservation[passengers_attrname] = {
                x: {'key': next(index)} for x in range(passengers_count)
            }

    return {'air_reservation': air_reservation}


BOOK_URL = 'http://api.pilot.ua/air_reservations'
DEEPLINK_TEMPLATE = 'http://air.pilot.ua/%s/air_reservations/%s/edit/'


def book(order_data):
    r = requests.post(
        BOOK_URL,
        headers={'Content-Type': 'application/json'},
        data=order_data['air_reservation'],
    )
    r.raise_for_status()

    token = r.json()['token']
    lang = order_data.get('lang', 'uk')
    url = DEEPLINK_TEMPLATE % (lang, token)

    return url
