# -*- coding: utf-8 -*-
import logging
import urllib
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = logging.getLogger(__name__)

SEARCH_URL = 'https://crs.bookingcontrolsolutions.com/Api/YandexSearch'
SERVICE_CLASS_MAP = {'economy': 'E', 'business': 'B'}
SERVICE_REVERSE_CLASS_MAP = {v: k for k, v in SERVICE_CLASS_MAP.items()}


def validate_query(q):
    q.validate_klass(SERVICE_CLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml)

    return variants


def get_data(tracker, q):
    query_params = {
        'arrival': q.iata_to,
        'departure': q.iata_from,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'adult': q.passengers.get('adults', 0),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0),
        'class': SERVICE_CLASS_MAP[q.klass],
    }

    if q.date_backward:
        query_params.update({
            'date_backward': q.date_backward.strftime('%Y-%m-%d'),
        })

    r = tracker.wrap_request(
        requests.get,
        SEARCH_URL,
        params=urllib.urlencode(query_params),
        verify=False,
    )

    return r.content


def parse_segments(fare_tags):
    segments = []

    for fare_tag in fare_tags:
        segments.append(parse_flight(fare_tag))

    return segments


def parse_response(xml):
    tree = etree.fromstring(xml)

    variants = []

    for v_tag in sleep_every(tree.xpath('variant')):

        v = Variant()

        v.forward.segments = parse_segments(v_tag.xpath('route_forward'))
        v.backward.segments = parse_segments(v_tag.xpath('route_backward'))

        fare_tag = v_tag.xpath('fare')[0]

        v.klass = SERVICE_REVERSE_CLASS_MAP[fare_tag.get('class')]

        v.charter = fare_tag.get('charter') == 'true'
        v.block = fare_tag.get('block') == 'true'

        v.tariff = Price(
            float(fare_tag.get('value')),
            fare_tag.get('currency')
        )

        v.url = v_tag.get('url')
        v.order_data = {'url': v.url}

        v.raw_tarrifs = {f.number: f.raw_tariffs for f in v.all_segments if hasattr(v, 'raw_tariffs')}

        variants.append(v)

    return variants


def parse_flight(f_tag):
    f = IATAFlight()

    f.station_from_iata = f_tag.get('departure_airport_code')
    f.station_to_iata = f_tag.get('arrival_airport_code')

    FMT = '%Y-%m-%d %H:%M'
    f.local_departure = datetime.strptime(f_tag.get('departure_datetime'), FMT)
    f.local_arrival = datetime.strptime(f_tag.get('arrival_datetime'), FMT)

    f.company_iata = f_tag.get('company_code')
    f.number = f.company_iata + ' ' + f_tag.get('route_code')

    tariff_adult = f_tag.get('tariff_adult')
    tariff_child = f_tag.get('tariff_child')
    tariff_infant = f_tag.get('tariff_infant')

    if any([tariff_adult, tariff_child, tariff_infant]):
        f.raw_tariffs = {
            'adult': tariff_adult,
            'child': tariff_child,
            'infant': tariff_infant,
        }

    return f
