# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import datetime
from itertools import product

import requests
import ujson
from typing import Any, Dict, List, Optional

from travel.avia.ticket_daemon.ticket_daemon.api.query import Query, QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import update_query_string, url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

POBEDA_API_URL = 'http://fares.pobeda.aero/FareSearch'
POBEDA_CLIENT_ID = partner_secret_storage.get(
    importer_name='pobeda', namespace='CLIENT_ID'
)

POBEDA_DEEPLINK_BASE_URL = 'https://pobeda.aero/ExternalSearch.aspx'

CURRENCIES_MAP = {'ru': 'RUB', 'ua': 'UAH', 'tr': 'TRY', 'com': 'EUR', 'kz': 'RUB'}
CULTURES_MAP = {'ru': 'ru-RU', 'ua': 'ru-RU', 'tr': 'en-US', 'com': 'en-US', 'kz': 'ru-RU'}
URL_TRACKER = {'utm_source': 'yandex'}
PREMIUM_BUNDLE_CODE = 'PRE1'
PLUS_BUNDLE_CODE = 'PLS2'

Bundle = namedtuple('Bundle', ('fare_code', 'baggage', 'price_calculator', 'deeplink_params_builder',))


class BundleCode(object):
    BASE = 'BASE'
    PLUS = 'PLUS'
    PREMIUM = 'PREMIUM'
    BASE_AND_BAGGAGE10 = 'BASE_AND_BAGGAGE10'
    BASE_AND_BAGGAGE20 = 'BASE_AND_BAGGAGE20'


AVAILABLE_BUNDLES = {
    BundleCode.BASE: Bundle(
        fare_code='BASE',
        baggage=Baggage.from_partner(0),
        price_calculator=lambda route: route['FarePrice'],
        deeplink_params_builder=lambda q: {'bundleCode': 'STD'},
    ),
    BundleCode.PLUS: Bundle(
        fare_code='PLS2',
        baggage=Baggage.from_partner(weight=20),
        price_calculator=lambda route: route['FarePrice'] + route['PlusChargePrice'],
        deeplink_params_builder=lambda q: {'bundleCode': 'PLS2'},
    ),
    BundleCode.PREMIUM: Bundle(
        fare_code='PRE1',
        baggage=Baggage.from_partner(weight=20),
        price_calculator=lambda route: route['FarePrice'] + route['PremiumChargePrice'],
        deeplink_params_builder=lambda q: {'bundleCode': 'PRE1'},
    ),
    BundleCode.BASE_AND_BAGGAGE10: Bundle(
        fare_code='BAG10',
        baggage=Baggage.from_partner(weight=10),
        price_calculator=lambda route: route['FarePrice'] + route['Baggage10KgPrice'],
        deeplink_params_builder=lambda q: {'bundleCode': 'STD', 'setSSR': 'B10:{}'.format(non_infants_count(q))},
    ),
    BundleCode.BASE_AND_BAGGAGE20: Bundle(
        fare_code='BAG20',
        baggage=Baggage.from_partner(weight=20),
        price_calculator=lambda route: route['FarePrice'] + route['Baggage20KgPrice'],
        deeplink_params_builder=lambda q: {'bundleCode': 'STD', 'setSSR': 'B20:{}'.format(non_infants_count(q))},
    ),
}

BUNDLE_CODES = (
    BundleCode.BASE,
    BundleCode.PLUS,
    BundleCode.PREMIUM,
    BundleCode.BASE_AND_BAGGAGE10,
    BundleCode.BASE_AND_BAGGAGE20,
)

PobedaRoute = namedtuple('PobedaRoute', 'route_string local_departure_time local_arrival_time flight_number')

logger = logging.getLogger(__name__)


def validate_query(q):
    if q.klass != 'economy':
        raise QueryIsNotValid('Only economy requests allowed')


@QueryTracker.init_query
def query(tracker, q):
    json_content = get_data(tracker, q)
    variants = generate_variants(json_content, q)
    return variants


def get_data(tracker, q):
    url = update_query_string(POBEDA_API_URL, build_search_params(q))
    r = tracker.wrap_request(
        requests.get,
        url,
    )

    return r.json()


def build_search_params(q):
    # type: (Query) -> Dict
    request_parameters = [{
        'Route': [make_route(q.station_iatas_from, q.station_iatas_to)],
        'BeginDate': q.date_forward.strftime('%Y-%m-%d'),
        'EndDate': q.date_forward.strftime('%Y-%m-%d'),
        'CurrencyCode': CURRENCIES_MAP[q.national_version],
        'AdultCount': q.passengers.get('adults', 0),
        'ChildCount': q.passengers.get('children', 0),
        'InfantCount': q.passengers.get('infants', 0),
        'APIVersion': 4.7,
    }]

    if q.date_backward:
        request_parameters.append({
            'Route': [make_route(q.station_iatas_to, q.station_iatas_from)],
            'BeginDate': q.date_backward.strftime('%Y-%m-%d'),
            'EndDate': q.date_backward.strftime('%Y-%m-%d'),
            'CurrencyCode': CURRENCIES_MAP[q.national_version],
            'AdultCount': q.passengers.get('adults', 0),
            'ChildCount': q.passengers.get('children', 0),
            'InfantCount': q.passengers.get('infants', 0),
            'APIVersion': 4.7,
        })

    return {
        'requestParameters': ujson.dumps(request_parameters),
        'clientId': POBEDA_CLIENT_ID,
    }


def make_route(iatas_from, iatas_to):
    # type: (List[str], List[str]) -> str
    return '{iata_from}-{iata_to}'.format(
        iata_from=','.join(iatas_from),
        iata_to=','.join(iatas_to),
    )


def generate_variants(results, q):
    variants = []
    for route_pair in sleep_every(generate_route_pairs(results)):
        variants.extend(get_variants_for_route_pair(route_pair, q))
    return variants


def generate_route_pairs(results):
    if len(results) == 1:
        return ((r, None) for r in get_raw_routes(results[0]))

    return (
        (forward, backward) for forward, backward in product(get_raw_routes(results[0]), get_raw_routes(results[1]))
        if forward['FromIata'] == backward['ToIata'] and forward['ToIata'] == backward['FromIata']
    )


def get_raw_routes(response):
    raw_routes = []

    for result in response['Results']:
        if 'Value' not in result or result['Value'] is None:
            logger.warning('Wrong response: %s', response)
            continue
        for route in result['Value']:
            route['FromIata'] = route['Route'][:3]
            route['ToIata'] = route['Route'][-3:]
            raw_routes.append(route)

    return raw_routes


def get_variants_for_route_pair(route_pair, q):
    variants = []
    for bundle_code in BUNDLE_CODES:
        variant = build_variant_for_route_pair(route_pair, q, AVAILABLE_BUNDLES[bundle_code])
        if variant:
            variants.append(variant)
    return variants


def build_variant_for_route_pair(route_pair, q, bundle):
    # type: (List[Dict[str, Any]], Query, Bundle) -> Optional[Variant]
    try:
        passenger_price = sum(bundle.price_calculator(r) for r in route_pair if r)
        infant_passenger_price = sum(r['InfantFarePrice'] for r in route_pair if r)
    except Exception:
        # Если не можем расчитать цену для бандла, то пропускаем его
        return None

    v = Variant()
    v.forward.segments = list(get_segments(q.importer.flight_fabric, route_pair[0], bundle))
    if q.date_backward:
        v.backward.segments = list(get_segments(q.importer.flight_fabric, route_pair[1], bundle))

    v.klass = q.klass
    v.order_data = make_order_data(v, q, bundle)
    v.tariff = Price(
        passenger_price * non_infants_count(q) + infant_passenger_price * q.passengers.get('infants', 0),
        currency=route_pair[0]['CurrencyCode'],
    )

    return v


def get_segments(flight_fabric, raw_route, bundle):
    routes = [
        PobedaRoute(*t) for t in zip(
            raw_route['Route'].split('/'),
            raw_route['LocalDepartureTime'].split('/'),
            raw_route['LocalArrivalTime'].split('/'),
            raw_route['FlightNumber'].split('/'),
        )
    ]

    for route in routes:
        yield flight_fabric.create(
            station_from_iata=route.route_string[:3],
            station_to_iata=route.route_string[3:],
            local_departure=parse_datetime(route.local_departure_time),
            local_arrival=parse_datetime(route.local_arrival_time),
            company_iata=route.flight_number[:2],
            pure_number=route.flight_number[2:],
            baggage=bundle.baggage,
            fare_code=bundle.fare_code,
        )


# https://st.yandex-team.ru/RASPTICKETDAEMON-258
def non_infants_count(q):
    return sum([
        q.passengers.get('adults', 0),
        q.passengers.get('children', 0),
    ])


def make_order_data(v, q, bundle):
    deeplink_params = make_deeplink_params(v, q, bundle)
    return {
        'url': url_complement_missing(POBEDA_DEEPLINK_BASE_URL, deeplink_params),
    }


def make_deeplink_params(v, q, bundle):
    deeplink_params = {
        'fromStation': v.forward.segments[0].station_from_iata,
        'toStation': v.forward.segments[-1].station_to_iata,
        'beginDate': v.forward.segments[0].local_departure.strftime('%d-%m-%Y'),
        'adultCount': q.passengers.get('adults', 0),
        'childrenCount': q.passengers.get('children', 0) or None,
        'infantCount': q.passengers.get('infants', 0) or None,
        'marketType': 'RoundTrip' if q.date_backward else 'OneWay',
        'beginFlight': '/'.join([
            f.number.replace(' ', '') for f in v.forward.segments
        ]),
        'culture': CULTURES_MAP.get(q.national_version),
    }
    deeplink_params = skip_None_values(deeplink_params)

    if q.date_backward:
        deeplink_params['endDate'] = v.backward.segments[-1].local_departure.strftime('%d-%m-%Y')
        deeplink_params['endFlight'] = '/'.join([
            f.number.replace(' ', '') for f in v.backward.segments
        ])

    deeplink_params.update(URL_TRACKER)
    deeplink_params.update(bundle.deeplink_params_builder(q))

    return deeplink_params


def parse_datetime(raw_datetime):
    return datetime.strptime(
        raw_datetime,
        '%Y-%m-%dT%H:%M:%S'
    )


def book(order_data):
    return order_data['url']
