# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

POSOSHOK_URL = "http://api.pososhok.ru/pososhok/unixml/"


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = list(parse_response(xml, q))

    return variants


def get_data(tracker, q):
    params = {
        'command': 'search',
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'begin': q.iata_from,
        'end': q.iata_to,
        'departure': q.date_forward.strftime("%Y-%m-%d"),
        'partner': 'yar',
        'class': q.klass.upper()
    }

    if q.date_backward:
        params['return'] = q.date_backward.strftime("%Y-%m-%d")

    r = tracker.wrap_request(
        requests.get,
        POSOSHOK_URL,
        params=params
    )

    return r.content


def parse_segment(segment, flight_fabric):
    airline = segment.find('airline')
    time_format = '%Y-%m-%dT%H:%M:%S'
    return flight_fabric.create(
        company_iata=airline.find('code').text,
        fare_code=segment.find('farebasis').text,
        station_from_iata=segment.find('begin-location').find('code').text,
        station_to_iata=segment.find('end-location').find('code').text,
        local_departure=datetime.strptime(segment.find('departure').text, time_format),
        local_arrival=datetime.strptime(segment.find('arrival').text, time_format),
        pure_number=segment.find('flight').text,
    )


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    for variant in sleep_every(tree.xpath('.//variant')):
        v = Variant()

        # суммируем цены на взрослых, детей и младенцев
        v.tariff = Price(sum(q.passengers.get(price.tag, 0) * float(price.text)
                             for price in variant.find('price').getchildren()))

        # надо выделить из сегментов группы по признаку "начальный"
        segments_groups = []
        for segment in variant.find('segments').getchildren():
            if segment.find('starting').text == 'true':
                segments_groups.append([])

            segments_groups[-1].append(segment)

        # если получилась одна группа, значит перелет в одну сторону
        # если две, то первая - это перелет туда, а вторая - обратно
        # если другое количество групп, то это какая-то чушь
        if not 1 <= len(segments_groups) <= 2:
            log.error(u'Вариант с неправильным количеством групп сегментов')
            continue

        segments_forward, segments_backward = segments_groups \
            if len(segments_groups) == 2 else (segments_groups[0], [])

        v.forward.segments = [
            parse_segment(segment, q.importer.flight_fabric) for segment in segments_forward
        ]

        v.backward.segments = [
            parse_segment(segment, q.importer.flight_fabric) for segment in segments_backward
        ]

        v.url = variant.find('url').text

        v.klass = q.klass

        v.order_data = {'url': v.url}

        yield v
