# -*- coding: utf-8 -*-
import logging
from typing import Any, Iterable

from travel.avia.library.python.sirena_client.django import sirena_client_factory
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.lib.sirena import SirenaFetcher, SirenaVariant
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

logger = logging.getLogger(__name__)

CLIENT_ID = 2979
COMPANY_CODE = 'WZ'
BOOK_URL = 'https://webapi.flyredwings.com/online/?currencyCodeSelectedByUser=RUB&lang=ru#/passengers'


@QueryTracker.init_query
def query(tracker, q):
    # type: (QueryTracker, Query)->Iterable[Iterable[Variant]]
    logger.info('Query %s', q)

    fetcher = SirenaFetcher(sirena_client_factory(CLIENT_ID))
    logger.info('Getting variants for %s', q)
    variants = fetcher.get_variants(
        query=q,
        company_iata=COMPANY_CODE,
        deeplink=deeplink,
        on_finish=tracker.set_query_time_from_timer,
    )
    logger.info('Got variants iterator for %s', q)
    return variants


def deeplink(q, variant):
    # type: (Any, SirenaVariant)->Any
    params = {
        'segmentsCount': sum(len(j.flights) for j in variant.joints),
        'actualCurrency': 'RUB',
        'lang': 'ru',  # it's the only valid option for now
        'count-aaa': q.adults,
        'count-rbg': q.children,
        'count-rmg': q.infants,
        'direct-only': 0,
        'search-engine': 'yandex',
        'searchPeriod': 0,
    }
    i = 0
    last_joint_id = -1
    for joint_id, joint in enumerate(variant.joints, 1):
        for seq_number, flight in enumerate(joint.flights):
            seg_params = {
                'joint-id': joint_id,
                'carrier': flight.company,
                'flight': flight.num,
                'date': flight.departure.strftime('%d.%m.%Y'),
                'origin-city-code': flight.origin,
                'destination-city-code': flight.destination,
                'class': flight.class_,
                'subclass': flight.subclass,
            }
            if i > 0:
                del seg_params['subclass']
            for k, v in seg_params.items():
                if v is not None:
                    params['{}[{}]'.format(k, i)] = v
            if last_joint_id != joint_id:
                params['{}[{}]'.format('requestedBrands', i)] = variant.brand
                last_joint_id = joint_id
            i += 1
    if not variant.brand:
        logger.error('Variant with segment without brand: %r', params)
    return {
        'url': BOOK_URL,
        'params': params,
    }


def book(order_data):
    return {'url': order_data['url'], 'post_data': order_data['params']}
