# -*- coding: utf-8 -*-
import hashlib
from datetime import datetime
from logging import getLogger
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage, BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.library.python.ticket_daemon.date import get_utc_now
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)
SINDBAD_URL = 'https://xml.sindbad.ru/api/search/'
SINDBAD_LOGIN = 'yandex'
SINDBAD_PRIVATE_KEY = partner_secret_storage.get(
    importer_name='sinbad2', namespace='PASSWORD'
)


KLASS_MAP = {'economy': 'E', 'business': 'B'}
URL_TRACKER = {'utm_source': 'yandex.avia'}

NO_BAGGAGE = frozenset(['NO', 'CHK', 'NIL'])


def validate_query(q):
    q.validate_klass(KLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    book_url_template = tree.get('bookUrlTemplate')
    m_book_url_template = tree.get('mobBookUrlTemplate')

    variants = []

    for trip in sleep_every(tree.xpath('//trip')):
        v = Variant()

        trip_id = trip.get('id')
        v.url = url_complement_missing(
            book_url_template.replace('TRIP_ID', trip_id), URL_TRACKER)

        v.m_url = url_complement_missing(
            m_book_url_template.replace('TRIP_ID', trip_id), URL_TRACKER)

        v.klass = q.klass
        v.tariff = Price(float(
            (
                trip.xpath('price/cc') or
                trip.xpath('price/cash') or
                trip.xpath('price/qiwi')
            )[0].text
        ))

        segments_out = trip.xpath('out')
        segments_in = trip.xpath('in')

        if segments_out:
            v.forward.segments = parse_flights(segments_out[0], q.importer.flight_fabric)

        if segments_in:
            v.backward.segments = parse_flights(segments_in[0], q.importer.flight_fabric)

        v.order_data = {
            'url': v.url,
            'm_url': v.m_url,
        }

        variants.append(v)

    log.debug('Variants count: %s', len(variants))

    return variants


def get_baggage(segment):
    baggage_info = segment.findtext('BagInfo')
    if baggage_info in NO_BAGGAGE:
        return Baggage.from_partner(included=False)
    return baggage_parser.parse_from_string(baggage_info)


@pipe(list)
def parse_flights(tree_element, flight_fabric):
    for segment in tree_element.xpath('seg'):
        takeoff = segment.xpath('takeoff')[0]
        landing = segment.xpath('landing')[0]
        route = segment.xpath('flight')[0]
        yield flight_fabric.create(
            station_from_iata=takeoff.get('tp'),
            station_to_iata=landing.get('tp'),
            local_departure=datetime.strptime(
                takeoff.get('date') + takeoff.get('time'), '%Y-%m-%d%H:%M'
            ),
            local_arrival=datetime.strptime(
                landing.get('date') + landing.get('time'), '%Y-%m-%d%H:%M'
            ),
            company_iata=route.get('carrier'),
            pure_number=route.get('number'),
            baggage=get_baggage(segment),
        )


def get_data(tracker, q):
    # @see http://trac.sindbad.ru/wiki/ru/API/search
    params = {}

    params['src'] = q.iata_from.encode('utf-8')
    params['dst'] = q.iata_to.encode('utf-8')
    params['out'] = q.date_forward.strftime('%Y-%m-%d')

    if q.date_backward:
        params['in'] = q.date_backward.strftime('%Y-%m-%d')

    params['kind'] = "RT" if q.date_backward else "OW"
    params['adt'] = q.passengers.get('adults', 0)
    params['cnn'] = q.passengers.get('children', 0)
    params['inf'] = q.passengers.get('infants', 0)
    params['class'] = KLASS_MAP[q.klass]
    params['token'] = 'yandex'

    params['d'] = 'N'

    r = tracker.wrap_request(
        requests.post,
        SINDBAD_URL,
        headers={
            'Content-Type': 'application/x-www-form-urlencoded; charset=utf-8',
        },
        auth=requests.auth.HTTPBasicAuth(
            SINDBAD_LOGIN,
            sindbad_password(),
        ),
        data=urlencode(params)
    )

    return r.content


def sindbad_password():
    m = hashlib.md5()
    m.update(''.join([
        'sindbad', get_utc_now().strftime('%Y-%m-%d'), SINDBAD_PRIVATE_KEY
    ]))

    return m.hexdigest()
