# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

URL = 'https://partner-yandex.superkassa.net/api/partner/search.php'
HASH = partner_secret_storage.get(importer_name='superkassa2', namespace='HASH')
HASH_KZ = partner_secret_storage.get(importer_name='superkassa2', namespace='HASH_KZ')
LOGIN = 'yandex'
PASSWORD = partner_secret_storage.get(
    importer_name='superkassa2', namespace='PASSWORD'
)


@QueryTracker.init_query
def query(tracker, q):
    if q.klass not in ['economy', 'business']:
        log.warning('Class not in: economy, business')
        return []

    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        URL,
        auth=(LOGIN, PASSWORD),
        params=build_search_query(q)
    )

    return r.content


def build_search_query(q):
    params = {
        'from': q.iata_from,
        'to': q.iata_to,
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'cabin': {'economy': 'Y', 'business': 'C'}[q.klass],
        'hash': HASH if q.national_version != 'kz' else HASH_KZ,
    }
    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')
    return params


def parse_response(xml, q):
    parser = etree.XMLParser(recover=True)
    tree = etree.fromstring(xml, parser=parser)

    variants = []

    for e in sleep_every(tree.xpath('variant')):
        v = Variant()

        try:
            segments = e.xpath('segment')

            v.forward.segments = parse_flights(segments[0], q.importer.flight_fabric)
            has_backward = bool(q.date_backward and len(segments) > 1)
            if has_backward:
                v.backward.segments = parse_flights(segments[1], q.importer.flight_fabric)

            cabins = {ftag.findtext('cabin')
                      for segment_tag in segments[:1 + has_backward]
                      for ftag in segment_tag.xpath('flight')}
            if len(cabins) != 1:
                continue

            v.klass = {'Y': 'economy', 'C': 'business'}[cabins.pop()]

            v.tariff = Price(float(e.findtext('price')),
                             e.findtext('currency').upper())
            v.order_data = {'url': e.findtext('url')}

            v.charter = e.findtext('isCharter') == 'true'

        except Exception:
            log.error('Error while parse variant')
            continue

        variants.append(v)

    return variants


def parse_flights(segment_tag, flight_fabric):
    flights = []

    for ftag in segment_tag.xpath('flight'):
        flights.append(flight_fabric.create(
            station_from_iata=ftag.findtext('departure'),
            station_to_iata=ftag.findtext('arrival'),

            local_departure=datetime.strptime(
                ftag.findtext('departureDate') + ftag.findtext('departureTime'),
                '%Y-%m-%d%H:%M'
            ),
            local_arrival=datetime.strptime(
                ftag.findtext('arrivalDate') + ftag.findtext('arrivalTime'),
                '%Y-%m-%d%H:%M'
            ),
            company_iata=ftag.findtext('operatingCarrier'),
            pure_number=ftag.findtext('number'),
            baggage=baggage_parser.parse_from_string(ftag.findtext('baggage')),
            selfconnect=ftag.findtext('is_selfconnect') == 'true',
            fare_code=ftag.findtext('fare_basis'),

        ))

    return flights
