# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

CODE = 'yandexru'
PASSPHRASE = partner_secret_storage.get(
    importer_name='supersaver', namespace='PASSWORD'
)

SUPERSAVER_URL_BY_NV = {
    'ru': 'http://ws.supersaver.ru/ws/ultrasearch/1.0/yandexru',
    'tr': 'http://ws.tr.gotogate.com/ws/ultrasearch/1.0/yandexru',
    'ua': 'http://ws.ua.gotogate.com/ws/ultrasearch/1.0/yandexru',
    'com': 'http://ws.ie.gotogate.com/ws/ultrasearch/1.0/yandexru',
}

NSMAP = {}
KLASS_MAP = {
    'economy': 'Y',  # Y = Economy class cabin
    'business': 'C',  # C = Business class cabin
    'first': 'F',  # F = First class cabin
    # todo: что делать с промежуточными классами?
    # P = Premium First class cabin
    # J = Premium Business class cabin
    # S = Premium Economy class cabin
}


def validate_query(q):
    q.validate_passengers(count=9)
    if q.national_version not in SUPERSAVER_URL_BY_NV:
        raise QueryIsNotValid('National version %s is not allowed', q.national_version)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = list(_parse_response(xml, q))
    return variants


def build_aviasearch_params(q):
    return {
        'forward_date': q.date_forward.strftime('%Y-%m-%d'),
        'return_date': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or None,
        'iata_from': q.iata_from,
        'iata_to': q.iata_to,
        'code': CODE,
        'passphrase': PASSPHRASE,

        'adults': range(1, 1 + q.passengers.get('adults', 0)),
        'children': range(1, 1 + q.passengers.get('children', 0)),
        'infants': range(1, 1 + q.passengers.get('infants', 0)),

        'cabin': KLASS_MAP[q.klass],
    }


def get_data(tracker, q):
    query_xml = loader.render_to_string(
        'partners/supersaver.xml', build_aviasearch_params(q)
    )

    r = tracker.wrap_request(
        requests.post,
        SUPERSAVER_URL_BY_NV.get(q.national_version),
        data=query_xml.encode('utf-8'),
    )

    return r.content


def _handle_unknown_fault(tree, fault_code_element):
    error_parts = [u'supersaver faultcode: {}'.format(fault_code_element.text)]
    if tree.xpath('//faultstring'):
        fault_string = tree.xpath('//faultstring')[0].text
        error_parts.append(u'faultstring: {}'.format(fault_string))
    if tree.xpath('//faultactor'):
        fault_actor = tree.xpath('//faultstring')[0].text
        error_parts.append(u'faultactor: {}'.format(fault_actor))
    raise Exception(' '.join(error_parts))


def _parse_response(xml, q):
    tree = etree.fromstring(xml)

    fault_code_elements = tree.xpath('//faultcode')
    if fault_code_elements:
        fault_code_element = fault_code_elements[0]
        if fault_code_element.text == 'noCombinedResult':
            return
        elif fault_code_element.text == 'ws.request':
            log.warning(u'Wrong request error: %s', xml.decode('utf-8'))
            return
        log.exception(u'Partner error: %s', xml.decode('utf-8'))
        _handle_unknown_fault(tree, fault_code_element)

    NSMAP['ns'] = tree.nsmap[None]  # Не очень красивый хак с ns для пустого нэймспэйса

    try:
        offers = tree.xpath('//ns:Offers', namespaces=NSMAP)[0]
        flight_offers = tree.xpath('//ns:FlightOffers', namespaces=NSMAP)[0]
    except IndexError:
        return

    currency = offers.find('ns:Currency', namespaces=NSMAP).get('code')
    for offer in sleep_every(offers.xpath('ns:Offer', namespaces=NSMAP)):
        v = Variant()
        v.klass = q.klass
        price = float(offer.findtext('ns:Price', namespaces=NSMAP))

        url = offer.get('url')
        v.url = url
        v.order_data = {'url': v.url}
        v.tariff = Price(price, currency=currency)

        flight_ref = offer.find('ns:FlightOfferRef', namespaces=NSMAP).get('ref')
        flight_offer = flight_offers.find('ns:FlightOffer[@id="%s"]' % flight_ref, namespaces=NSMAP)

        # selling_carrier = flight_offer.get('sellingCarrier')  # Todo: оставил на будущее
        v.forward.segments = parse_segments(
            q.importer.flight_fabric,
            flight_offer.xpath('ns:SegmentGroup[@type="out"]/ns:Segment', namespaces=NSMAP)
        )
        if q.date_backward:
            v.backward.segments = parse_segments(
                q.importer.flight_fabric,
                flight_offer.xpath('ns:SegmentGroup[@type="in"]/ns:Segment', namespaces=NSMAP)
            )

        yield v


@pipe(list)
def parse_segments(flight_fabric, segments):
    for segment in segments:
        _from = segment.find('ns:From', namespaces=NSMAP)
        _to = segment.find('ns:To', namespaces=NSMAP)
        yield flight_fabric.create(
            company_iata=segment.get('carrier'),
            pure_number=segment.get('flight'),
            station_from_iata=_from.get('iata'),
            station_to_iata=_to.get('iata'),
            local_departure=_parse_date(_from),
            local_arrival=_parse_date(_to),
            baggage=None
        )


def _parse_date(element):
    year, month, day = map(int, element.findtext('ns:Date', namespaces=NSMAP).split('-'))
    hour, minute = map(int, element.findtext('ns:Time', namespaces=NSMAP).split(':'))
    return datetime(year, month, day, hour, minute)
