# -*- coding: utf-8 -*-
import json
import logging

import requests

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise, pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import parse_datetime_without_seconds

log = logging.getLogger(__name__)

URL_TRACKER = {
    'utm_source': 'yandex.ru'
}

KLASS_NAME2CODE = {'economy': 'E', 'business': 'B'}
KLASS_CODE2NAME = {v: k for k, v in KLASS_NAME2CODE.items()}


def validate_query(q):
    q.validate_klass(KLASS_NAME2CODE)


@QueryTracker.init_query
def query(tracker, q):
    return parse_response(get_data(tracker, q), q)


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.post,
        'http://meta.svyaznoy.travel/multireq/site/metasearch/',
        params=build_query_params(q)
    )
    data = json.loads(r.content)
    if 'err' in data:
        return []
    return data


@pipe(dict)
def build_query_params(q):
    yield 'source', 'yandex.ru'
    yield 'route', ''.join([
        q.date_forward.strftime('%d%m'),
        q.iata_from,
        q.iata_to,
        q.date_backward and q.date_backward.strftime('%d%m') or '',
    ])
    yield 'cs', KLASS_NAME2CODE[q.klass]
    yield 'ad', q.passengers.get('adults', 1)
    yield 'cn', q.passengers.get('children', 0)
    yield 'in', q.passengers.get('infants', 0)


def parse_response(data, q):
    variants = []
    for raw_variant in sleep_every(data):
        try:
            v = parse_variant(q, raw_variant)
            if v:
                variants.append(v)
        except Exception:
            log.exception('Error in parse_variant')
    return variants


def get_baggage(raw):
    try:
        is_baggage = raw.get('is_baggage')
        if is_baggage is None:
            return Baggage.from_partner()
        elif is_baggage is False:
            return Baggage.from_partner(0)
        elif is_baggage is True:
            raw_baggage_info = raw.get('baggage_info')
            if raw_baggage_info is not None:
                baggage_info = raw_baggage_info.split('.')
                if len(baggage_info) == 2:
                    count, desc = baggage_info
                    if desc == 'KG':
                        return Baggage.from_partner(weight=count)
                    elif desc == 'PC':
                        return Baggage.from_partner(pieces=count)
                    else:
                        log.warning('Unknown baggage format %s', raw.get('baggage_info'))
            return Baggage.from_partner(included=True)
        else:
            log.error('Invalid "is_baggage": %s', is_baggage)
    except Exception, exc:
        log.error('Baggage parsing exception: %s on element %s', exc, raw)

    return Baggage.from_partner()


def parse_variant(q, raw):
    log.debug('Parse variant: %r', raw)

    v = Variant()

    baggage = get_baggage(raw)

    v.forward.segments = parse_flights(raw['route_forward'],
                                       q.importer.flight_fabric, baggage)
    if q.date_backward and 'route_backward' not in raw:
        return
    if q.date_backward:
        v.backward.segments = parse_flights(
            raw['route_backward'], q.importer.flight_fabric, baggage
        )

    v.charter = bool(int(raw['fare']['charter']))

    segments_klasses = list(set(s.klass for s in v.all_segments))
    if len(segments_klasses) != 1 or segments_klasses[0] != q.klass:
        log.debug('Service classes does not match: %r != %r', segments_klasses, q.klass)
        return
    v.klass = q.klass

    v.tariff = Price(float(raw['total_sum']), 'RUR')
    v.url = raw['url']
    v.order_data = {'url': v.url}

    return v


@elementwise
def parse_flights(raw, flight_fabric, baggage):
    return flight_fabric.create(
        company_iata=raw['company_code'],
        number=u'{} {}'.format(raw['company_code'], raw['route_code'][:-2]),
        fare_code=raw.get('farecode'),
        station_from_iata=raw['departure_airport_code'],
        station_to_iata=raw['arrival_airport_code'],
        local_departure=parse_datetime_without_seconds(raw['departure_datetime']),
        local_arrival=parse_datetime_without_seconds(raw['arrival_datetime']),
        klass=KLASS_CODE2NAME[raw['class']],
        baggage=baggage,
    )
