# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from datetime import datetime
from logging import getLogger

import requests
from django.conf import settings
from django.utils.http import urlencode
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import async_sleep, sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

TICKETS_UA_AVIA_URL = 'http://avia.api.tickets.ua/avia.php'
TICKETS_UA_AVIA_KEY = {
    'ru': partner_secret_storage.get(
        importer_name='ticketsua2', namespace='RUPASSWORD'
    ),
    'ua': partner_secret_storage.get(
        importer_name='ticketsua2', namespace='UAPASSWORD'
    ),
    'kz': partner_secret_storage.get(
        importer_name='ticketsua2', namespace='KZPASSWORD'
    ),
    'tr': partner_secret_storage.get(
        importer_name='ticketsua2', namespace='TRPASSWORD'
    ),
}
TICKETS_UA_REFID = {
    'ru': 171,
    'ua': 170,
    'kz': 498,
    'tr': 342,
}

TICKETS_UA_DRIVERS = {
    'ru': 'yandex',
    'ua': 'xml',
    'kz': 'xml_kz',
    'tr': 'xml_com_tr',
}

KLASS_MAP = {'economy': 'E', 'business': 'B'}


def validate_query(q):
    q.validate_klass(KLASS_MAP)
    q.validate_passengers(count=9)
    if q.national_version not in TICKETS_UA_AVIA_KEY:
        raise QueryIsNotValid(
            'No TICKETS_UA_AVIA_KEY for national_version [%s]' % q.national_version)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        TICKETS_UA_AVIA_URL,
        params=build_query_params(q)
    )

    return r.content


def build_query_params(q):
    params = [
        ('key', TICKETS_UA_AVIA_KEY.get(q.national_version, TICKETS_UA_AVIA_KEY['ru'])),
        ('action', 'search'),
        ('driver', TICKETS_UA_DRIVERS.get(q.national_version, TICKETS_UA_DRIVERS['ru'])),
        ('destinations[0][departure]', q.iata_from),
        ('destinations[0][arrival]', q.iata_to),
        ('destinations[0][date]', q.date_forward.strftime('%d-%m-%Y')),
    ]

    if q.date_backward:
        params += [
            ('destinations[1][departure]', q.iata_to),
            ('destinations[1][arrival]', q.iata_from),
            ('destinations[1][date]', q.date_backward.strftime('%d-%m-%Y')),
        ]

    params += [
        ('adt', q.passengers.get('adults', 0)),
        ('chd', q.passengers.get('children', 0)),
        ('inf', q.passengers.get('infants', 0)),
    ]

    params += [('service_class', KLASS_MAP[q.klass])]

    return params


def process_ticketsua_errors(q, errors):
    errors_texts = [
        u'%s:%s' % (ecode, etext)
        for ecode, etext in [(e.get('code'), e.text) for e in errors]
        # RASPTICKETDAEMON-551
        # Не спамить в логи об отсутствии вариантов у ticketsua
        if ecode not in ['66002']
    ]
    if errors_texts:
        log.error(u"%s %s", q.qid_msg, '|'.join(errors_texts))


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    async_sleep(0)

    # Ошибка: <?xml version="1.0" encoding="utf-8"?><RESULTS><ERRORS><ERROR code="N">N</ERROR></ERRORS></RESULTS>
    errors = tree.xpath('//ERROR')

    if len(errors):
        process_ticketsua_errors(q, errors)
        return []

    needed_currency = settings.AVIA_NATIONAL_CURRENCIES.get(q.national_version, 'RUR')
    refid = TICKETS_UA_REFID.get(q.national_version, TICKETS_UA_REFID['ru'])
    variants = []

    url = tree.findtext('REDIRECT_URL')
    url = u"{}&{}".format(
        url,
        urlencode({
            'class': KLASS_MAP[q.klass],
            'currency': needed_currency,
            'act': 'book',
            'refid': refid,
        })
    )

    for e in sleep_every(tree.xpath('/RESULTS/ITEMS/ITEM')):
        v = Variant()

        # Пример варианта:
        #
        # <RESULTS>
        #     <REDIRECT_URL>
        #       http://avia.tickets.ru/preloader?StartAirp1Code=MOW&EndAirp1Code=KGD&Date1=20-12-2016&adt=1&chd=0&inf=0&hlp=0&class=E&st=20.09.2016_18:17
        #     </REDIRECT_URL>
        #     <CURRENCY>RUR</CURRENCY>
        #     <ITEMS>
        #         <ITEM pay_currency="RUR" amount="2460.00" hash="6f312601e1f73c4738e5c637538a9d6b_U6167|:*161218560_90">
        #             <FARE_INFO>
        #                 <TOTAL_AMOUNT code="UAH" tariff_adult="976.00" tariff_child="0.00" tariff_infant="0.00">976.00</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="USD" tariff_adult="37.89" tariff_child="0.00" tariff_infant="0.00">37.89</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="EUR" tariff_adult="33.96" tariff_child="0.00" tariff_infant="0.00">33.96</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="RUR" tariff_adult="2460.00" tariff_child="0.00" tariff_infant="0.00">2460.00</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="PLN" tariff_adult="146.42" tariff_child="0.00" tariff_infant="0.00">146.42</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="AZN" tariff_adult="62.50" tariff_child="0.00" tariff_infant="0.00">62.50</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="KZT" tariff_adult="12864.00" tariff_child="0.00" tariff_infant="0.00">12864.00</TOTAL_AMOUNT>
        #                 <TOTAL_AMOUNT code="GBP" tariff_adult="28.99" tariff_child="0.00" tariff_infant="0.00">28.99</TOTAL_AMOUNT>
        #             </FARE_INFO>
        #             <TRIP_PART duration="115">
        #                 <SEGMENT>
        #                     <DEPARTURE city="MOW" airport="DME" terminal="">20.12.2016 16:05</DEPARTURE>
        #                     <ARRIVAL city="KGD" airport="KGD" terminal="">20.12.2016 17:00</ARRIVAL>
        #                     <FLIGHT aircraft="Airbus A319" service_class="economy" supplier_code="U6">167</FLIGHT>
        #                     <FARE_CODE>OPROW</FARE_CODE>
        #                 </SEGMENT>
        #             </TRIP_PART>
        #       </ITEM>

        # Не имеем контроля над классами. В составных перелетах идут смешанные.
        v.klass = q.klass

        amount_elems = e.xpath('.//FARE_INFO/TOTAL_AMOUNT[@code="%s"][1]' % needed_currency)

        if amount_elems:
            amount_elem = amount_elems[0]
            amount = amount_elem.text
            tariff_adult = amount_elem.attrib.get('tariff_adult')
            tariff_child = amount_elem.attrib.get('tariff_child')
            tariff_infant = amount_elem.attrib.get('tariff_infant')

            if any([tariff_adult, tariff_child, tariff_infant]):
                v.raw_tariffs = {
                    'adult': tariff_adult,
                    'child': tariff_child,
                    'infant': tariff_infant,
                }

            log.debug('Found needed_currency %s [%s]', needed_currency, amount)
        else:
            log.debug('Not found needed_currency %s', needed_currency)
            continue

        v.tariff = Price(float(amount), needed_currency)

        v.url = u"{}&{}".format(
            url,
            urlencode({
                'r_hash': e.get('hash')
            })
        )

        parts = e.xpath('TRIP_PART')
        if len(parts) > 0:
            segments_out = parts[0]
            if len(segments_out) > 0:
                v.forward.segments = list(parse_flights(segments_out, q.importer.flight_fabric))

                if len(parts) > 1:
                    segments_in = parts[1]
                    if len(segments_in) > 0:
                        v.backward.segments = list(parse_flights(segments_in, q.importer.flight_fabric))

        v.order_data = {'url': v.url}
        variants.append(v)

    return variants


def parse_datetime(raw_date):
    """
    Example '21.01.2018 13:50' for raw_date
    """
    return datetime(
        int(raw_date[6:10]), int(raw_date[3:5]), int(raw_date[0:2]),
        int(raw_date[11:13]), int(raw_date[14:16])
    )


def parse_flights(segments, flight_fabric):
    for seg in segments:
        # Пример элемента:
        # <SEGMENT>
        #     <DEPARTURE city="SVX" airport="SVX" terminal="">03.04.2012 19:50</DEPARTURE>
        #     <ARRIVAL city="MOW" airport="DME" terminal="">03.04.2012 20:20</ARRIVAL>
        #     <FLIGHT aircraft="Airbus Industrie A319" service_class="economy" supplier_code="S7">56</FLIGHT>
        #     <FARE_CODE>NVUR</FARE_CODE>
        # </SEGMENT>

        departure = seg.find('DEPARTURE')
        arrival = seg.find('ARRIVAL')
        flight = seg.find('FLIGHT')
        yield flight_fabric.create(
            station_from_iata=departure.get('airport'),
            station_to_iata=arrival.get('airport'),
            local_departure=parse_datetime(departure.text),
            local_arrival=parse_datetime(arrival.text),
            company_iata=flight.get('supplier_code'),
            pure_number=flight.text,
            baggage=baggage_parser.parse_from_string(flight.get('baggage')),
            fare_code=seg.findtext("FARE_CODE"),
        )
