# -*- coding: utf-8 -*-
import logging
import urllib
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = logging.getLogger(__name__)

CLIENT_ID = 36
CLIENT_KEY = 4
PASSWORD = partner_secret_storage.get(importer_name='tinkoff', namespace='PASSWORD')
SEARCH_URL = 'http://api.aviakassa.ru'
SERVICE_CLASS_MAP = {'economy': 'economy', 'business': 'business'}
FARE_CLASS_MAP = {'E': 'economy', 'B': 'business'}

URL_TRACKER = {
    'utm_source': 'Tinkoff_yandex',
    'utm_medium': 'cpc',
}


def validate_query(q):
    q.validate_klass(SERVICE_CLASS_MAP)
    q.validate_passengers(count=9)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    query_params = {
        'action': 'avia_meta',
        'client_key': CLIENT_KEY,
        'ClientId': CLIENT_ID,
        'password': PASSWORD,
        'adult': q.passengers.get('adults', 0),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0),
        'service_class': SERVICE_CLASS_MAP[q.klass],
        'segments[0][from]': q.iata_from,
        'segments[0][to]': q.iata_to,
        'segments[0][date]': q.date_forward.strftime('%d.%m.%Y'),
    }

    if q.date_backward:
        query_params.update({
            'segments[1][from]': q.iata_to,
            'segments[1][to]': q.iata_from,
            'segments[1][date]': q.date_backward.strftime('%d.%m.%Y')
        })

    r = tracker.wrap_request(
        requests.get,
        SEARCH_URL,
        params=urllib.urlencode(query_params)
    )

    return r.content


def make_segments(fare_tags, query):
    forward_segments = []
    backward_segments = []

    for fare_tag in fare_tags:
        segment = parse_flight(fare_tag, query.importer.flight_fabric)
        route_index = fare_tag.get('route_index')

        if route_index == '0':
            forward_segments.append(segment)

        elif route_index == '1':
            backward_segments.append(segment)

        else:
            raise BadRouteIndex

    return forward_segments, backward_segments


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    variants = []

    for v_tag in sleep_every(tree.xpath('variant')):
        v = Variant()
        fare_tag = v_tag.xpath('fare')[0]

        try:
            v.forward.segments, v.backward.segments = make_segments(v_tag.xpath('route'), q)

        except BadRouteIndex:
            continue

        v.klass = FARE_CLASS_MAP[fare_tag.get('class')]

        v.charter = fare_tag.get('charter') == '1'
        v.block = fare_tag.get('block') == 'true'

        v.tariff = Price(float(fare_tag.get('value')), 'RUR')

        v.url = url_complement_missing(
            urllib.unquote(v_tag.get('url')), URL_TRACKER
        )

        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


def get_baggage(fare_tag):
    try:
        pieces = fare_tag.get('pieces_of_luggage')
        if pieces is None:
            return Baggage.from_partner()
        elif pieces == '0':
            return Baggage.from_partner(pieces=int(pieces))
        else:
            weight = fare_tag.get('luggage_weight')
            return Baggage.from_partner(pieces=int(pieces), weight=int(weight))

    except Exception, exc:
        log.info('Baggage parsing exception: {} on element {}'.format(exc, fare_tag))
        return Baggage.from_partner()


def parse_flight(f_tag, flight_fabric):
    FMT = '%Y-%m-%d %H:%M'

    return flight_fabric.create(
        local_departure=datetime.strptime(f_tag.get('departure_datetime'), FMT),
        local_arrival=datetime.strptime(f_tag.get('arrival_datetime'), FMT),
        station_from_iata=f_tag.get('departure_airport_code'),
        station_to_iata=f_tag.get('arrival_airport_code'),
        company_iata=f_tag.get('company_code'),
        pure_number=f_tag.get('route_code'),
        baggage=get_baggage(f_tag),
    )


class BadRouteIndex(Exception):
    pass
