# -*- coding: utf-8 -*-
import logging
from datetime import datetime
from functools import partial

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker


log = logging.getLogger(__name__)

SEARCH_URL = 'https://api-travel.tinkoff.ru/api/v1/metasearch/yandex/search'

SERVICE_CLASS_MAP = {'economy': 'Y', 'business': 'C'}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def _query_params(q):
    query_params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'adult': q.passengers.get('adults', 0),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0),
        'class': SERVICE_CLASS_MAP[q.klass],
    }
    if q.date_backward:
        query_params['date_backward'] = q.date_backward.strftime('%Y-%m-%d')
    return query_params


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get, SEARCH_URL,
        params=_query_params(q)
    )
    return r.content


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    variants = []

    for v_tag in sleep_every(tree.xpath('variant')):
        v = Variant()
        fare_tag = v_tag.xpath('fare')[0]

        baggage = get_baggage(fare_tag)
        parse_flights = partial(
            parse_flight,
            flight_fabric=q.importer.flight_fabric, baggage=baggage
        )
        v.forward.segments = map(parse_flights, v_tag.xpath('route_forward'))

        if q.date_backward:
            v.backward.segments = map(parse_flights, v_tag.xpath('route_backward'))

        # v.klass = FARE_CLASS_MAP[fare_tag.get('class')] # TODO для эконома присылают M и Y
        v.klass = q.klass
        v.charter = fare_tag.get('charter') == 'true'

        v.tariff = Price(float(fare_tag.get('value')), fare_tag.get('currency'))

        v.url = v_tag.get('url')
        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


def get_baggage(fare_tag):
    try:
        pieces = fare_tag.get('pieces_of_luggage')
        weight = fare_tag.get('luggage_weight')

        baggage_info = {}
        if weight:
            baggage_info['weight'] = int(weight)
        if pieces:
            baggage_info['pieces'] = int(pieces)
        return Baggage.from_partner(**baggage_info)

    except Exception, exc:
        log.info('Baggage parsing exception: {} on element {}'.format(exc, fare_tag))
        return Baggage.from_partner()


def parse_flight(f_tag, flight_fabric, baggage):
    FMT = '%Y-%m-%d %H:%M'

    return flight_fabric.create(
        local_departure=datetime.strptime(f_tag.get('departure_datetime'), FMT),
        local_arrival=datetime.strptime(f_tag.get('arrival_datetime'), FMT),
        station_from_iata=f_tag.get('departure_airport_code'),
        station_to_iata=f_tag.get('arrival_airport_code'),
        company_iata=f_tag.get('company_code'),
        pure_number=f_tag.get('route_code'),
        baggage=baggage,
    )
