# -*- coding: utf-8 -*-
from collections import namedtuple

import requests
from django.utils.http import urlencode
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import parse_datetime_without_seconds

PartnerId = namedtuple('PartnerId', ('country', 'desktop', 'xml', 'mobile'))

PARTNER_IDS = {
    'es': PartnerId('ES', 'x742zbf', 'xmx742t', 'mbx742t'),
    'com': PartnerId('IE', 'x742zbfIE', 'xmx742tIE', 'mbx742tIE'),
    'ru': PartnerId('RU', 'x742zbfRU', 'xmx742tRU', 'mbx742tRU'),
    'tr': PartnerId('TR', 'x742zbfTR', 'xmx742tTR', 'mbx742tTR'),
    'kz': PartnerId('KZ', 'x742zbfKZ', 'xmx742tKZ', 'mbx742tKZ'),
    'ua': PartnerId('UA', 'x742zbfUA', 'xmx742tUA', 'mbx742tUA'),
}

DEEPLINK = 'http://dl.travelgenio.com/Results.aspx?{}'
API_URL = 'http://xml.travelgenio.com/ResultsXML.aspx'
KLASS_MAP = {'economy': 'Y', 'first': 'F', 'business': 'C'}


@QueryTracker.init_query
def query(tracker, q):
    search_params = get_search_params(q)
    r = tracker.wrap_request(requests.get, API_URL, params=search_params)
    variants = parse_response(r.content, q, search_params)
    return variants


def get_search_params(q):
    """
    :type q: ticket_daemon.api.query.Query
    """
    dt_format = '%d-%m-%Y'
    partner_id = PARTNER_IDS.get(q.national_version, PARTNER_IDS['ru'])
    params = {
        'Brand': 'TG',  # TG / T2B Brand (Travelgenio or Travel2Be)
        'Country': partner_id.country,
        'Orig': q.iata_from,
        'Dest': q.iata_to,
        'AD': q.passengers.get('adults', 0),
        'CH': q.passengers.get('children', 0),
        'INF': q.passengers.get('infants', 0),
        'DepDate': q.date_forward.strftime(dt_format),
        'RetDate': None,
        'Class': KLASS_MAP[q.klass],
        'DirectOnly': False,
        'OW': not bool(q.date_backward),
        'LowCost': True,
        'IncludeOptionCode': True,
        'Partner': partner_id.xml,
        'Extradata': True,
    }
    if q.date_backward:
        params['RetDate'] = q.date_backward.strftime(dt_format)
    return params


def build_prefix_deeplinks(q, search_params):
    params = search_params.copy()
    params.pop('Partner', None)
    partner_id = PARTNER_IDS.get(q.national_version, PARTNER_IDS['ru'])
    return {
        'url': DEEPLINK.format(
            urlencode(dict(Partner=partner_id.desktop, **params))),
        'm_url': DEEPLINK.format(
            urlencode(dict(Partner=partner_id.mobile, **params))),
    }


def get_baggage(variant):
    _type = variant.findtext('FreeBagUnitCode')
    quantity = int(variant.findtext('FreeBagQuantity'))
    if _type == 'K':
        return Baggage.from_partner(weight=quantity)
    elif _type == 'N':
        return Baggage.from_partner(pieces=quantity)
    return Baggage.from_partner()


@pipe(list)
def parse_response(xml, q, search_params):
    tree = etree.fromstring(xml)
    prefix_deeplinks = build_prefix_deeplinks(q, search_params)

    for variant in sleep_every(tree.xpath('./Options/Option')):
        # RASPTICKETS-18746: если ничего не найдено, партнер присылает один пустой вариант
        if len(variant) == 0:
            continue

        v = Variant()
        v.klass = q.klass

        dep_stops = variant.findtext('DepStops')
        dep_stops_count = dep_stops.count('|') + 1 if dep_stops else 1
        segments = parse_segments(
            variant.xpath('./Segments/Segment'), q.importer.flight_fabric
        )
        v.forward.segments = segments[:dep_stops_count]
        if q.date_backward:
            v.backward.segments = segments[dep_stops_count:]
        v.tariff = Price(
            float(variant.findtext('TotalAfterDiscounts')),
            currency=variant.findtext('CurrID')
        )
        additional_params = {
            'OptionCode': variant.findtext('OptionCode')
        }
        v.order_data = {
            'url': prefix_deeplinks['url'] + u'&' + urlencode(
                additional_params
            ),
            'm_url': prefix_deeplinks['m_url'] + u'&' + urlencode(
                additional_params
            )
        }
        yield v


@pipe(list)
def parse_segments(segments, flight_fabric):
    for segment in segments:
        yield flight_fabric.create(
            company_iata=segment.findtext('AirlineID'),
            pure_number=segment.findtext('FlightNum'),
            station_from_iata=segment.findtext('Origin'),
            station_to_iata=segment.findtext('Destination'),
            local_departure=parse_datetime_without_seconds(
                segment.findtext('DepDateTime')
            ),
            local_arrival=parse_datetime_without_seconds(
                segment.findtext('ArrDateTime')
            ),
            baggage=get_baggage(segment),
        )
