# -*- coding: utf-8 -*-
import json
from logging import getLogger

import requests
from django.utils.http import urlencode

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, min_class
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import parse_datetime_without_seconds

log = getLogger(__name__)

UTM_MARKS = {
    'ru': {
        'utm_source': 'Avia.yandex_RU',
        'utm_medium': 'affiliate',
        'utm_campaign': '496',
    },
    'ua': {
        'utm_source': 'Avia.yandex_UA',
        'utm_medium': 'affiliate',
        'utm_campaign': '13429',
    },
    'kz': {
        'utm_source': 'Avia.yandex_KZ',
        'utm_medium': 'affiliate',
        'utm_campaign': '19124',
    },
    'tr': {
        'utm_source': 'ticket.yandex',
        'utm_medium': 'referral',
        'utm_campaign': 'avia.yandex.tr',
    },
}

SEARCH_URL = 'http://ws.pamediakopes.gr/json/Search'

TRIP_RU_BRANDS = {
    'ru': 'trip.ru',
    'ua': 'trip.ua',
    'kz': 'trip.kz',
    'tr': 'mytrip.com/tr'
}

TRIP_RU_CURRENCIES = {
    'ru': 'RUB',
    'ua': 'UAH',
    'kz': 'KZT',
    'tr': 'TRY',
}

TRIP_RU_LANGS = {
    'ru': 'ru',
    'ua': 'uk',
    'kz': 'ru',
    'tr': 'tr',
}

TRIP_RU_COUNTRIES = {
    'ru': 'RU',
    'ua': 'UA',
    'kz': 'KZ',
    'tr': 'TR',
}

AIDS = {
    'ru': 496,
    'ua': 13429,
    'kz': 19124,
    'tr': 15132,
}

KLASS_MAP = {'economy': 'Y', 'business': 'C', 'first': 'F'}
KLASS_REVERSE_MAP = {'Y': 'economy', 'C': 'business', 'F': 'first'}


def validate_query(q):
    q.validate_passengers(adults=8, children=7)


@QueryTracker.init_query
def query(tracker, q):
    json_data = get_data(tracker, q)

    variants = list(parse_response(json_data, q))

    return variants


def parse_response(json_data, q):
    part_deep_link = create_part_deep_link(q)
    for j in sleep_every(json_data['Journeys']):
        v = Variant()

        v.tariff = Price(
            j['TotalPrice'],
            TRIP_RU_CURRENCIES.get(q.national_version)
        )

        v.forward.segments = list(parse_flights(q.importer.flight_fabric, j.get('OutTrip')))
        if q.date_backward:
            v.backward.segments = list(parse_flights(q.importer.flight_fabric, j.get('ReturnTrip')))

        v.klass = min_class(v.klasses)

        v.order_data = {
            'url': create_deep_link(part_deep_link, j, v)

            # https://st.yandex-team.ru/RASPTICKETS-11311
            # 'url': url_complement_missing(
            #     create_deep_link(part_deep_link, j, v),
            #     UTM_MARKS.get(q.national_version, UTM_MARKS['ru'])
            # )
        }

        # если класс варианта не соответствует запросу, то пропускаем
        if v.klass == q.klass:
            yield v


def create_part_deep_link(q):
    params = make_params(q)

    if 'ReturnDate' not in params:
        params['ReturnDate'] = ''

    params['DirectOnly'] = 'False'
    params['Language'] = TRIP_RU_LANGS.get(q.national_version)

    return 'https://www.{brand}/flights/meta/landing?{params}'.format(
        brand=TRIP_RU_BRANDS.get(q.national_version),
        params=urlencode(params)
    )


def create_deep_link(part_deep_link, j, v):
    return part_deep_link + '&' + urlencode({
        'RedirectToken': j['DeepLinkToken'],
        'TotalPrice': j['TotalPrice'],
        'FlightNumbers': ','.join(
            [s.number.replace(' ', '-') for s in v.all_segments]
        )
    })


def get_baggage(flight):
    try:
        if not flight['FreeBaggageAllowance']:
            return Baggage.from_partner()

        included = flight['BaggageIncluded']
        baggage_allowance = flight['FreeBaggageAllowance']['Adult']
        weight_unit = baggage_allowance.get('WeightUnit')
        weight = baggage_allowance.get('Weight') if weight_unit == 'kg' else None
        pieces = baggage_allowance.get('Pieces')
        return Baggage.from_partner(pieces=pieces, weight=weight, included=included)
    except Exception as exc:
        log.error('Baggage parsing exception: %s on element %s', exc, flight, exc_info=True)
        return Baggage.from_partner()


def parse_flights(flight_fabric, segments_data):
    segments = segments_data['Flights']

    if not segments:
        return

    for flight in segments:
        yield flight_fabric.create(
            local_departure=parse_datetime_without_seconds(flight['StartDate']),
            local_arrival=parse_datetime_without_seconds(flight['EndDate']),
            station_from_iata=flight['Origin'],
            station_to_iata=flight['Destination'],
            company_iata=flight['MarketingCarrier'],
            pure_number=flight['FlightNumber'],
            klass=KLASS_REVERSE_MAP.get(flight['SeatClassCode']),
            baggage=get_baggage(flight)
        )


def make_params(q):
    iso_8601_format = '%Y-%m-%d'

    request_data = {
        'DepartureDate': q.date_forward.strftime(iso_8601_format),
        'Origin': q.iata_from,
        'Destination': q.iata_to,
        'SeatClassCode': KLASS_MAP.get(q.klass),
        'Adults': q.passengers.get('adults'),
        'Children': q.passengers.get('children'),
        'Infants': q.passengers.get('infants'),
        'Country': TRIP_RU_COUNTRIES.get(q.national_version),
        'Currency': TRIP_RU_CURRENCIES.get(q.national_version),
        'AffiliateId': AIDS.get(q.national_version),
        'Brand': TRIP_RU_BRANDS[q.national_version]
    }

    if q.date_backward:
        request_data['ReturnDate'] = q.date_backward.strftime(iso_8601_format)

    return request_data


def get_data(tracker, q):
    request_data = make_params(q)

    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        data=json.dumps(request_data),
        headers={
            'Content-Type': 'application/json',
        },
    )

    return r.json()
