# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

TRIPSTA_LOGIN = 'schedule'
TRIPSTA_PASSWD = partner_secret_storage.get(
    importer_name='tripsa5', namespace='PASSWORD'
)
TRIPSTA_URL = {
    'ru': 'http://www.tripsta.ru/flight/api/soap',
    'ua': 'http://www.tripsta.com.ua/flight/api/soap',
    'tr': 'http://www.tripsta.com.tr/flight/api/soap',
}
URL_TRACKER = {
    'utm_source': 'yandex_schedule',
    'utm_medium': 'metasearch',
    'utm_campaign': 'flights',
}


KLASS_MAP = {'economy': 'Y', 'business': 'C', 'first': 'F'}
KLASS_MAP_REVERSE = {
    'Y': 'economy',
    'W': 'economy',
    'C': 'business',
    'F': 'first',
}


def validate_query(q):
    q.validate_klass(KLASS_MAP)

    if not (hasattr(q, 'iata_real_from') and hasattr(q, 'iata_real_to')):
        raise QueryIsNotValid('From or To airports dont have IATA code')


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    params = {
        'username': TRIPSTA_LOGIN,
        'password': TRIPSTA_PASSWD,
        'dep': q.iata_from,
        'arr': q.iata_to,
        'obDate': q.date_forward,
        'ibDate': q.date_backward,
        'isRoundtrip': q.date_backward and 1 or 0,
        'passengersAdult': q.passengers.get('adults', 0),
        'passengersChild': q.passengers.get('children', 0),
        'passengersInfant': q.passengers.get('infants', 0),
        'class': KLASS_MAP[q.klass],
    }

    query_xml = loader.render_to_string(
        'partners/tripsta5.xml', params).strip()

    tripsta_soap_url = TRIPSTA_URL.get(
        q.national_version,
        TRIPSTA_URL['ru']
    )

    r = tracker.wrap_request(
        requests.post,
        tripsta_soap_url,
        headers={
            'Content-Type': 'text/xml; charset=utf-8',
            'Connection': 'Keep-Alive',
            'SOAPAction': tripsta_soap_url,
        },
        data=query_xml.encode('utf-8')
    )

    return r.content


@pipe(list)
def parse_response(xml, q):
    tree = etree.fromstring(xml)

    for e in sleep_every(tree.xpath('//return/item')):
        v = Variant()

        currencies = {
            p.text
            for p in e.findall('passengers/item/total/currencyCode')
        }

        if len(currencies) != 1:
            continue

        fare = sum([
            (float(p.findtext('total/amount')) * float(p.findtext('count')))
            for p in e.xpath('passengers/item')
        ])

        curerncy = currencies.pop()
        curerncy = {'RUB': 'RUR'}.get(curerncy, curerncy)
        v.tariff = Price(fare, curerncy)

        v.klass = q.klass

        outbound = e.find('legs/item[type="OB"]')
        inbound = e.find('legs/item[type="IB"]')

        if outbound:
            v.forward.segments = parse_flights(outbound, q.importer.flight_fabric)

        if inbound:
            v.backward.segments = parse_flights(inbound, q.importer.flight_fabric)

        # https://st.yandex-team.ru/RASPTICKETS-11311
        # url_tracker = URL_TRACKER.copy()
        #
        # if v.forward.segments:
        #     url_tracker['utm_term'] = '%s_%s' % (
        #         v.forward.segments[0].station_from_iata,
        #         v.forward.segments[-1].station_to_iata
        #     )
        #
        # v.url = url_complement_missing(
        #     e.findtext('url'),
        #     url_tracker
        # )

        v.url = e.findtext('url')
        v.order_data = {'url': v.url}

        yield v


def get_baggage(baggage):
    if baggage == 'true':
        return Baggage.from_partner(included=True)
    elif baggage == 'false':
        return Baggage.from_partner(included=False)
    else:
        return Baggage.from_partner()


def parse_datetime(raw_year, raw_month, raw_day, raw_time):
    return datetime(
        int(raw_year), int(raw_month), int(raw_day),
        int(raw_time[0:2]), int(raw_time[3:5]),
    )


@pipe(list)
def parse_flights(segments, flight_fabric):
    baggage = get_baggage(segments.findtext('baggageIncluded'))
    for seg in segments.xpath('segments/item'):
        dep_datetime = parse_datetime(
            seg.findtext('depYear'),
            seg.findtext('depMonth'),
            seg.findtext('depDay'),
            seg.findtext('depTime')
        )

        arr_datetime = parse_datetime(
            seg.findtext('arrYear'),
            seg.findtext('arrMonth'),
            seg.findtext('arrDay'),
            seg.findtext('arrTime')
        )

        yield flight_fabric.create(
            local_departure=dep_datetime,
            local_arrival=arr_datetime,
            station_from_iata=seg.findtext('dep'),
            station_to_iata=seg.findtext('arr'),
            company_iata=seg.findtext('airlineCode'),
            pure_number=seg.findtext('flightNumber'),
            klass=KLASS_MAP_REVERSE.get(seg.findtext('cabinClass'), 'economy'),
            baggage=baggage,
        )
