# -*- coding: utf-8 -*-
import logging
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values

log = logging.getLogger(__name__)

SEARCH_URL = 'https://api.tripway.com/yandex/v1/avia/trips.xml'


def validate_query(q):
    q.validate_klass(['economy'])


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        data=skip_None_values({
            'departure': q.iata_from,
            'arrival': q.iata_to,
            'date_forward': q.date_forward.strftime('%Y-%m-%d'),
            'date_backward': (q.date_backward.strftime('%Y-%m-%d')
                              if q.date_backward else None),
            'class': {'economy': 'E', 'business': 'B'}[q.klass],
            'adult': q.passengers.get('adults', 0),
            'child': q.passengers.get('children', 0),
            'infant': q.passengers.get('infants', 0),
        })
    )

    return r.text.encode('utf-8')


def parse_response(xml, q):
    tree = etree.fromstring('<root>%s</root>' % xml)

    variants = []

    for v_tag in sleep_every(tree.xpath('variant')):
        v = Variant()

        v.forward.segments = parse_flights(v_tag.xpath('route_forward'))

        if q.date_backward:
            v.backward.segments = parse_flights(v_tag.xpath('route_backward'))

        fare_tag = v_tag.xpath('fare')[0]

        v.klass = {'E': 'economy', 'B': 'business'}[fare_tag.get('class')]

        currency = 'UAH'
        v.charter = fare_tag.get('charter') == 'true'
        v.block = fare_tag.get('block') == 'true'

        v.tariff = Price(float(fare_tag.get('value')), currency)

        v.url = v_tag.get('url')

        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


@elementwise
def parse_flights(f_tag):
    try:
        return _parse_flight(f_tag)

    except Exception:
        log.warning(u'f_tag: %r', f_tag)

        raise


def _parse_flight(f_tag):
    f = IATAFlight()

    f.station_from_iata = f_tag.get('departure_airport_code')
    f.station_to_iata = f_tag.get('arrival_airport_code')

    FMT = '%Y-%m-%d %H:%M'
    f.local_departure = datetime.strptime(f_tag.get('departure_datetime'), FMT)
    f.local_arrival = datetime.strptime(f_tag.get('arrival_datetime'), FMT)

    f.company_iata = f_tag.get('company_code')
    f.number = f.company_iata + ' ' + f_tag.get('route_code')

    return f
