# -*- coding: utf-8 -*-
import logging
from datetime import datetime

import requests

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, OperatingFlight, min_class
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.partners import add_param_to_url

log = logging.getLogger(__name__)

TUTU_SEARCH_URL = 'https://yapi.tutu.ru/search'
TUTU_PARTNER_CODE = 'tutu'
TUTU_LOGIN = 'yandex'
TUTU_PASSWORD = partner_secret_storage.get(importer_name=TUTU_PARTNER_CODE, namespace='PASSWORD')

# todo correct value
KLASS_MAP = {'economy': 'ECONOMY', 'business': 'BUSINESS', }
KLASS_REVERSE_MAP = {'ECONOMY': 'economy', 'BUSINESS': 'business', }


@QueryTracker.init_query
def query(tracker, q):
    json = get_data(tracker, q)
    variants = parse_response(json, q)
    return variants


def book(order_data):
    brand = order_data.get('avia_brand')
    if brand == TUTU_PARTNER_CODE:
        return add_param_to_url('unisearchquery', 'brand', order_data['url'])
    return order_data['url']


def _query_params(q):
    query_params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'class': KLASS_MAP[q.klass],
    }
    if q.date_backward:
        query_params['date_backward'] = q.date_backward.strftime('%Y-%m-%d')
    return query_params


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        TUTU_SEARCH_URL,
        params=_query_params(q),
        auth=requests.auth.HTTPBasicAuth(TUTU_LOGIN, TUTU_PASSWORD),
    )
    return r.json()


def parse_response(json, q):
    variants = []
    for variant in sleep_every(json.get('variant', [])):
        try:
            v = Variant()
            v.klass = q.klass

            v.forward.segments = [parse_flight(flight, q.importer.flight_fabric)
                                  for flight in variant.get('route_forward')]

            if q.date_backward:
                v.backward.segments = [parse_flight(flight, q.importer.flight_fabric)
                                       for flight in variant.get('route_backward')]

            price = variant.get('price')
            v.tariff = Price(
                float(price.get('value')),
                price.get('currency')
            )
            v.url = variant['url']
            v.order_data = {'url': v.url}

            v.klass = min_class(v.klasses)

            variants.append(v)

        except Exception as e:
            log.error('Error while parsing variant: %r', e)
    return variants


def parse_flight(f_tag, flight_fabric):
    FMT = '%Y-%m-%d %H:%M:%S'

    return flight_fabric.create(
        local_departure=datetime.strptime(f_tag.get('departure_datetime'), FMT),
        local_arrival=datetime.strptime(f_tag.get('arrival_datetime'), FMT),
        station_from_iata=f_tag.get('departure_airport_code'),
        station_to_iata=f_tag.get('arrival_airport_code'),
        company_iata=f_tag.get('marketing_carrier'),
        operating=OperatingFlight(
            company_iata=f_tag.get('operating_carrier'),
            number=f_tag.get('operating_flight_number'),
        ),
        number=f_tag.get('flight_number'),
        klass=KLASS_REVERSE_MAP[f_tag.get('class')],
        fare_code=f_tag.get('fare_code'),
        fare_family=f_tag.get('fare_family'),
        baggage=parse_baggage(f_tag),
        selfconnect=f_tag.get('selfconnect'),
        charter=f_tag.get('charter'),
    )


def parse_baggage(f_tag):
    luggage = f_tag.get('luggage')
    return Baggage.from_partner(luggage.get('pieces', 0), luggage.get('weight', 0), luggage.get('included', False))
