# -*- coding: utf-8 -*-
import re
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

UFS_PLANE_URL = 'https://ym.ufs-online.ru/webservices/avia/Metasearch/pricing/yandex'
UFS_PLANE_USER = 'YandexOOO'
UFS_PLANE_PASSWORD = partner_secret_storage.get(
    importer_name='ufs_plane2', namespace='PASSWORD'
)
BAGGAGE_FORMAT = re.compile(r'(\d+)\s?(KG|K|PC|N)', re.UNICODE)
COUNT_DESIGNATION = {'N', 'PC'}
WEIGHT_DESIGNATION = {'K', 'KG'}
EMPTY_BAGGAGE_DESIGNATION = {'NO', '0PC', '0 PC', u'БЕЗ БАГАЖА'}


def validate_query(q):
    q.validate_country_codes()


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = list(parse_response(xml, q))

    return variants


KLASS_MAP = {
    'economy': 'E',
    'business': 'B',
    'first': 'F',
}

REVERSE_KLASS_MAP = {v: k for k, v in KLASS_MAP.items()}


def get_data(tracker, q):
    params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'class': KLASS_MAP[q.klass],
        'AdultQuantity': q.passengers.get('adults', 0),
        'ChildQuantity': q.passengers.get('children', 0),
        'BabyQuantityNoPlace': q.passengers.get('infants', 0)
    }

    if q.date_backward:
        params['date_backward'] = q.date_backward.strftime('%Y-%m-%d')

    r = tracker.wrap_request(
        requests.get,
        UFS_PLANE_URL,
        params=params,
        auth=requests.auth.HTTPBasicAuth(
            UFS_PLANE_USER,
            UFS_PLANE_PASSWORD,
        )
    )

    return r.text.encode('utf-8')


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    for vtag in sleep_every(tree.xpath('//YandexStructures/variant')):
        try:
            yield parse_variant(q, vtag)
        except Exception, exc:
            log.warning(u'Ошибка разбора варианта %s', exc)


def parse_variant(q, vtag):
    v = Variant()

    fare = vtag.find('fare')

    v.tariff = Price(float(fare.get('value')), 'RUR')

    v.klass = REVERSE_KLASS_MAP[fare.get('class')]

    v.forward.segments = list(parse_flights(
        vtag.xpath('./route_forward'), fare, q.importer.flight_fabric))

    if q.date_backward:
        v.backward.segments = list(parse_flights(
            vtag.xpath('./route_backward'), fare, q.importer.flight_fabric))

    v.url = vtag.get('url')

    v.charter = vtag.get('route_type') == 'Charter'

    v.order_data = {
        'url': v.url,
    }

    return v


def get_baggage(route_tag, fare):
    try:
        baggage = route_tag.get('baggage')
        if baggage is None:
            return Baggage.from_partner()
        elif baggage in EMPTY_BAGGAGE_DESIGNATION:
            return Baggage.from_partner(included=False)

        count, desc = BAGGAGE_FORMAT.search(baggage).groups()
        if desc in COUNT_DESIGNATION:
            weight = fare.get('luggage_weight') or None
            if weight is not None:
                try:
                    weight_count, weight_desc = BAGGAGE_FORMAT.search(weight).groups()
                    weight = int(weight_count) if weight_desc in WEIGHT_DESIGNATION else None
                except:
                    log.info('Unknown weight format %s', weight)
            return Baggage.from_partner(pieces=int(count), weight=weight)
        elif desc in WEIGHT_DESIGNATION:

            pieces = fare.get('pieces_of_luggage') or None
            if pieces is not None:
                try:
                    pieces = int(pieces)
                except:
                    log.info('Unknown pieces_of_luggage format %s', pieces)
            return Baggage.from_partner(weight=int(count), pieces=pieces)
        else:
            log.info('Unknown baggage format %s', baggage)
            return Baggage.from_partner()

    except Exception, exc:
        log.info(
            'Baggage parsing exception: %r on route_tag %s fare_tag %s',
            exc,
            etree.tostring(route_tag, encoding='unicode'),
            etree.tostring(fare, encoding='unicode')
        )
    return Baggage.from_partner()


def parse_flights(routes_tags, fare, flight_fabric):
    for rtag in routes_tags:
        yield flight_fabric.create(
            local_departure=datetime.strptime(
                rtag.get('departure_datetime'), '%Y-%m-%d %H:%M'
            ),
            local_arrival=datetime.strptime(
                rtag.get('arrival_datetime'), '%Y-%m-%d %H:%M'
            ),
            company_iata=rtag.get('company_code'),
            station_from_iata=rtag.get('departure_airport_code'),
            station_to_iata=rtag.get('arrival_airport_code'),
            pure_number=rtag.get('route_code'),
            baggage=get_baggage(rtag, fare),
        )
