# -*- coding: utf-8 -*-
from logging import getLogger

import urlparse

from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.partners.sig import SigImporter, BUSINESS_CLASS, ECONOMY_CLASS
from travel.avia.ticket_daemon.ticket_daemon.settings import YANDEX_ENVIRONMENT_TYPE


importer_name = 'u6'

if YANDEX_ENVIRONMENT_TYPE == 'production':
    customer_id = 'Yandex_U6'
    search_url = 'https://sigak.tais.ru/SIG'
    password_suffix = 'PASSWORD'
else:
    customer_id = 'Yandex_U6_test'
    search_url = 'https://sigtest.tais.ru/SIG'
    password_suffix = 'TEST_PASSWORD'

UTMS = {
    'utm_source': 'yandex',
    'utm_medium': 'meta',
}


def get_tariff_codes(deeplink, logger):
    tariff_codes_forward = []
    tariff_codes_backward = []
    for key, value in sorted(urlparse.parse_qsl(urlparse.urlparse(deeplink).query)):
        if not key.startswith('FBA_'):
            continue
        if key.startswith('FBA_1_'):
            tariff_codes_forward.append(value)
        elif key.startswith('FBA_2_'):
            tariff_codes_backward.append(value)
    if not tariff_codes_forward:
        logger.warning('Couldn\'t extract tariff codes from deeplink: %s', deeplink)
    return tariff_codes_forward, tariff_codes_backward


sig_importer = SigImporter(
    customer_id=customer_id,
    search_url=search_url,
    api_password=partner_secret_storage.get(
        importer_name=importer_name, namespace=password_suffix
    ),
    utm=UTMS,
    logger=getLogger(__name__),
    tariff_codes_from_deeplink_extractor=get_tariff_codes,
)


@QueryTracker.init_query
def query(tracker, q):
    yandex_plus = YandexPlusPromo()
    return yandex_plus.fill_yandex_plus_promo(
        sig_importer.query(tracker, q, lambda v: not(q.klass == ECONOMY_CLASS and v.klass == BUSINESS_CLASS))
    )


def validate_query(q):
    sig_importer.validate_query(q)


class YandexPlusPromo(object):
    def order_data(self, variant):
        return {'total': min(3000, int(variant.tariff.value*0.03))}

    def fill_yandex_plus_promo(self, variants):
        for v in variants:
            v.order_data['yandex_plus_promo'] = self.order_data(v)
        return variants
