# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from logging import getLogger

from flask import Blueprint, request
from marshmallow import Schema, fields
from werkzeug.exceptions import BadRequest

from travel.avia.library.python.ticket_daemon.query_validation.validation import validate_query, ValidationError
from travel.avia.ticket_daemon.ticket_daemon.api import regionalization
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query, QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.api.result import set_partners_statuses, Statuses
from travel.avia.ticket_daemon.ticket_daemon.lib.query_partner_provider import query_partner_provider
from travel.avia.ticket_daemon.ticket_daemon.lib.timelines import QueryTimeLines
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.init_search_logger import init_search_logger
from travel.avia.ticket_daemon.ticket_daemon.views.helpers import (
    jsend_error, jsend_success, jsend_fail, fetch
)

init_query_routes = Blueprint('init query', __name__)
log = getLogger(__name__)


class PartnersListField(fields.List):
    def _serialize(self, value, attr, obj):
        raise NotImplementedError

    def _deserialize(self, value, attr, data):
        if not value:
            return []
        try:
            return list(set(value.split(',')))
        except Exception as e:
            log.warning('%r %r', e, value)
            self.fail('invalid')


class WizardCachesField(fields.List):
    def _serialize(self, value, attr, obj):
        raise NotImplementedError

    def _deserialize(self, value, attr, data):
        if not value:
            return []
        try:
            return list(set(value.split(',')))
        except Exception as e:
            log.warning('%r %r', e, value)
            self.fail('invalid')


class QueryPartnersPureSchema(Schema):
    qid = fields.String(required=True)  # qid, по которому запустится поиск
    base_qid = fields.String(required=False, missing=None)  # qid пользовательского запроса, который инициировал запросы для множественного поиска.

    ignore_cache = fields.Boolean(required=False)
    custom_store_time = fields.Integer(missing=0, required=False)
    wizard_caches = WizardCachesField(fields.String(), missing='', required=False)
    partner_codes = PartnersListField(fields.String(), missing='', required=False)
    test_id = fields.String(required=False, missing=None)


def _query_partners_pure(params):
    qid = params['qid']
    try:
        query = Query.from_qid(qid)
        query.meta['custom_store_time'] = params['custom_store_time']
        query.meta['wizard_caches'] = params['wizard_caches']
        query.meta['base_qid'] = params['base_qid']
        query.meta['test_id'] = params['test_id']
        query.meta['ignore_cache'] = params['ignore_cache']
    except QueryIsNotValid as e:
        raise BadRequest('{}: {}'.format(e, qid))

    query = QueryTimeLines.add_timelines(query)

    try:
        validate_query(query)
    except ValidationError as e:
        raise BadRequest(e.data)

    try:
        query.prepare_attrs_for_import()
    except QueryIsNotValid as e:
        raise BadRequest('{}: {}'.format(e, qid))

    partners_context = query_partner_provider.get_partners_for(
        query=query,
        only_partners=params['partner_codes'],
        ignore_cache=params['ignore_cache'],
        test_id=params['test_id']
    )

    _, skip_partners = regionalization.apply(query, partners_context.partners)
    partners_context.set_skip_partners(skip_partners)
    set_partners_statuses(query, skip_partners, Statuses.SKIP)
    query.partners = partners = partners_context.partners  # Todo, хорошо было бы не записывать partners в query
    if len(partners):
        fetch(query, partners_context)
        query.timeline.event(QueryTimeLines.events.spawned)
    init_search_logger.log(query, partners_context)

    return {
        'partners': [p.code for p in partners],
        'qid': query.id,
    }


@init_query_routes.route('/api/1.0/query_partners_pure/', methods=['GET'])
def query_partners_pure():
    try:
        params, errors = QueryPartnersPureSchema().load(request.values)
        if errors:
            log.warning('Params are not valid: %r', errors)
            raise BadRequest({'errors': errors})
        result = _query_partners_pure(params)

    except (ValueError, AssertionError) as e:
        return jsend_fail({'cause': repr(e)})
    except BadRequest as e:
        return jsend_fail({'cause': e.description})
    except Exception as e:
        log.exception(u'query_partners_pure')
        return jsend_error(repr(e))
    return jsend_success(result)
