# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import gevent
from flask import jsonify, request, Blueprint
from logging import getLogger

from requests import RequestException

from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import use_db, with_closing_db, logit
from travel.avia.ticket_daemon.ticket_daemon.api.redirect import (
    get_redirect_partner, fetch_redirect_data,
    get_partner_redirect_timeout, RedirectError
)
from travel.avia.ticket_daemon.ticket_daemon.lib.revise import register_redirect
from travel.avia.ticket_daemon.ticket_daemon.partners import PartnerBookNotFoundException

log = getLogger(__name__)
redirect_routes = Blueprint('redirect routes', __name__)


@redirect_routes.route('/api/1.0/cook_redirect/', methods=['POST'])
@use_db
def cook_redirect_1_0():
    order_data = request.json['order_data']
    additional_data = request.json.get('additional_data')

    try:
        partner = get_redirect_partner(order_data)
        timeout = get_partner_redirect_timeout(partner.code)

        redir_data = gevent.with_timeout(
            # Таймаут чуть меньше чем ждут нас
            timeout - 0.4 if timeout >= 2 else timeout * 0.8,
            fetch_redirect_data, partner, order_data,
        )

    except gevent.Timeout:
        log.warn(u'Redirector timeout: %r', order_data)

        return 'Redirector timeout', 408
    except PartnerBookNotFoundException as e:
        log.warn(
            u'Partner can not find redirect data: [%r] [%r]',
            e.partner_code, e.qid
        )

        return 'Partner can not find redirect data', 404
    except (RedirectError, RequestException) as e:
        log.exception(u'Partner redirect error: %r', e)

        return 'Redirector error', 400

    except Exception as e:
        log.exception(u'Redirector error: %r %r', e, order_data)

        return 'Redirector error', 500

    if not redir_data:
        log.error(u'Unknown redirector result: %r', order_data)
        return 'Unknown redirector result', 404

    # Запомнить редирект для проверки партнёрских цен
    try:
        user_info = request.json['user_info']
    except KeyError:
        log.warning('No user_info provided to cook_redirect_1_0')
    else:
        log.debug(
            'Data to register_redirect: %r %r %r %r',
            user_info, order_data, partner, redir_data
        )
        gevent.spawn(
            logit(log, with_closing_db(register_redirect)),
            user_info,
            order_data,
            partner,
            redir_data['url'], redir_data.get('post_data'),
            request.json['utm_source'],
            request.json['query_source'],
            additional_data,
        )

    return jsonify(redir_data)
