# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import os
from logging import getLogger

import gevent
from flask import Blueprint, jsonify
from flask import request, send_from_directory

from travel.avia.library.python.common.models.partner import Partner, DohopVendor
from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant

from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.flask_helpers import jsend_response
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer import Importer
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer_dialog import ImporterDialog
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import eat_exception, logit_with, with_closing_db, use_db
from travel.avia.ticket_daemon.ticket_daemon.lib import feature_flags
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_debugger import PartnerDebugger
from travel.avia.ticket_daemon.ticket_daemon.lib.timer import Timeline
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.timelines import QueryTimeLines

log = getLogger(__name__)
track_routes = Blueprint('track routes', __name__)


@eat_exception
@logit_with(log.warning)
def spawnable_fetch(q):
    # TODO This will be much more lightweight if already regionalized q.partners
    for importer in Importer.gen_for_partners(q.partners, q):
        gevent.spawn(logit_with(log.exception, with_closing_db(importer.do_import)))

    q.timeline.event(QueryTimeLines.events.spawned_importers, {
        'partners_count': len(q.partners),
    })


def query_params_to_qkey(query_params):
    return '_'.join(map(str, [
        query_params['point_from'], query_params['point_to'],
        query_params['date_forward'], query_params['date_backward'],
        query_params['klass'], query_params['passengers'].get('adults', 0),
        query_params['passengers'].get('children', 0),
        query_params['passengers'].get('infants', 0),
        query_params['national_version']]))


@track_routes.route('/api/1.0/track/start/', methods=['POST'])
@jsend_response
@use_db
def track_start():
    """Используется в админке, для тестирования партнеров"""
    d = request.json['query_params']
    q = Query.from_key(query_params_to_qkey(d), service=d['service'], lang=d['lang'])

    q = QueryTimeLines.add_timelines(q)

    p_code = request.json['p_code']
    partner = Partner.objects.get(code=p_code)
    q.partners = AmadeusMerchant.get_actual(national_version=d['national_version']) if partner.code == 'amadeus' else [partner]
    q.partners = DohopVendor.get_actual(national_version=d['national_version']) if partner.code == 'dohop' else [partner]

    q.timeline = Timeline('timeline.tracked').with_defaults({'qid': q.id})

    q.prepare_attrs_for_import()

    ImporterDialog.activate_query(q)
    if feature_flags.enable_new_partner_debugger():
        partner_debugger = PartnerDebugger()
        partner_debugger(q, partner)
    else:
        spawnable_fetch(q)

    importer_code = Importer.get_importer_code_for_partner(partner)

    return {'trid': ImporterDialog(q, importer_code).key}


@track_routes.route('/api/1.0/track/result/', methods=['GET'])
@jsend_response
@use_db
def track_result():
    """Используется в админке, для тестирования партнеров"""
    trid = request.values['trid']
    return ImporterDialog.get_results_by_key(trid)


@track_routes.route('/api/1.0/track/bad/', methods=['GET'])
@track_routes.route('/api/1.0/track/bad/<path:path>', methods=['GET'])
def bad_tracks(path=''):
    full_path = os.path.join(QueryTracker.TRACKS_PATH, 'bad', path)
    if os.path.isdir(full_path):
        return jsonify({
            'status': 'ok',
            'data': filter_lock_files(os.listdir(full_path))
        })
    return send_from_directory(os.path.join(QueryTracker.TRACKS_PATH, 'bad'), path)


def filter_lock_files(names):
    return [name for name in names if not name.endswith('.lock')]
