# -*- coding: utf-8 -*-
"""
Запускать:
    Y_PYTHON_ENTRY_POINT="travel.avia.ticket_daemon.tools.tickets_query_partner:main" /app/app
"""

import os
import sys
import logging

from copy import copy
from datetime import date, datetime, timedelta
from itertools import chain
from optparse import OptionParser, Option, OptionValueError
from subprocess import check_output
from traceback import format_exc

import django
django.setup()

from django.utils import translation

from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant
from travel.avia.library.python.common.models.geo import Point, Settlement, Station
from travel.avia.library.python.common.models.partner import DohopVendor, Partner
from travel.avia.ticket_daemon.ticket_daemon.api.flights import FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query, QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer import Importer, ModuleImporter
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_debugger import HumanVariantFormatter, find_failed_check
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

logging.basicConfig()
log = logging.getLogger(__name__)
log.addHandler(logging.StreamHandler(sys.stdout))

translation.activate('ru')


def show_results(variants):
    for j, v in enumerate(variants, start=1):
        print(j is not None and ('... (' + str(j) + ') ') or '') + '.' * 80
        print HumanVariantFormatter.format_variant(v)


def got_failure(failure):
    print failure


def partner_codes():
    return Partner.objects.filter(t_type__code='plane').values_list('code', flat=True)


def create_query_from_options(options):
    if options.query:
        q = options.query
        q.lang = options.lang
        q.service = options.service

    else:

        if not options.point_from or not options.point_to:
            print 'point_from, point_to must be provided'
            return

        date_forward = options.date_forward or date.today() + timedelta(days=7)
        date_backward = options.date_backward

        query_params = dict([
            ('point_from', options.point_from),
            ('point_to', options.point_to),
            ('date_forward', date_forward),
            ('date_backward', date_backward),
            ('klass', options.service_class),
            ('passengers', {
                k: getattr(options, k)
                for k in ('adults', 'children', 'infants')
            }),
            ('national_version', options.national_version or 'ru'),
            ('service', options.service),
            ('lang', options.lang)
        ])

        q = Query(**query_params)
        q = Query.from_key(q.key(), service=q.service, lang=q.lang)
    q.prepare_attrs_for_import()

    return q


def get_variants(q, partner):
    flight_fabric = FlightFabric()

    importer_code = Importer.get_importer_code_for_partner(partner)
    print 'Importer code:', importer_code
    importer_cls = Importer.get_importer_cls_by_code(importer_code)
    print 'Importer cls:', importer_cls.__name__
    importer = importer_cls(
        code=importer_code,
        partners=[partner],
        query=q,
        response_collector=None,
        flight_fabric=flight_fabric,
        big_beauty_collectors=None,
    )

    print 'Query module: {}'.format(
        ModuleImporter.import_module('travel.avia.ticket_daemon.ticket_daemon.partners.%s' % importer_code)
    )
    q = importer.q

    chunked_queryfun = importer.prepare_chunked_queryfun(importer_code, q)

    variants = list(chain.from_iterable(chunked_queryfun(q)))
    importer.assign_partners_to_variants(variants)

    variants = [
        v for v in variants
        if getattr(v, 'partner', None) and
        v.partner.code == partner.code
    ]

    for variant in variants:
        for segment in variant.all_segments:
            segment.complete(partner_code=partner.code, is_charter=variant.is_charter)

    print 'Flights:', len(flight_fabric.get_flights())

    return variants


def run(options, args):
    partner_secret_storage.recache()

    if not options.partner:
        print 'Specify partner'
        return
    partner = options.partner

    q = create_query_from_options(options)
    if not q:
        return

    q.trackers = {}
    q.need_store_tracks = True

    variants = []
    try:
        variants = get_variants(q, partner)

    except QueryIsNotValid as e:
        print 'Query is not valid', e

    except Exception as e:
        got_failure(format_exc())

    else:
        if options.show_results:
            for v in variants:
                v.what_bad = []
                find_failed_check(v, q)

            show_results(variants)

        print '.' * 80

    if q.trackers:
        tracker = q.trackers.values()[0]

        if tracker.store_place:
            if options.trace:

                print check_output(
                    'for j in $(find "{}" -type f | sort); do echo $j; zcat -f $j; echo; done'.format(tracker.store_place),
                    shell=True
                )

            else:
                if os.path.exists(tracker.store_place):
                    print 'Tracks in:\n{}'.format(tracker.store_place)
                    print check_output(['ls', '-d', tracker.store_place])

    else:
        print 'No traces!'

        print '.' * 89
        print 'Request params:'
        print u'partner: %s' % partner.code

        show_query_params(q)


def show_query_params(q):
    query_params = (
        ('point_from', q.point_from),
        ('point_to', q.point_to),
        ('date_forward', q.date_forward),
        ('date_backward', q.date_backward),
        ('klass', q.klass),
        ('passengers', q.passengers),
        ('national_version', q.national_version),
    )

    for j in query_params:
        print u'%s: %s' % j


def check_date(option, opt, value):
    for format in ['%d.%m.%Y', '%d-%m-%Y', '%Y.%m.%d', '%Y-%m-%d']:
        try:
            return datetime.strptime(value, format).date()
        except ValueError:
            pass

    raise OptionValueError(
        'option %s: invalid date value: %r' % (opt, value))


def check_point(option, opt, value):
    try:
        return Point.get_by_key(value)

    except Exception:
        pass

    value = value.lower()

    try:
        return Settlement.objects.get(iata__iexact=value)

    except Exception:
        pass

    try:
        return Settlement.objects.get(sirena_id__iexact=value)

    except Exception:
        pass

    try:
        return Station.objects.filter(code_set__system__code='iata'). \
            distinct().get(code_set__code__iexact=value)

    except Exception:
        pass

    try:
        return Station.objects.filter(code_set__system__code='sirena'). \
            distinct().get(code_set__code__iexact=value)

    except Exception:
        pass

    raise OptionValueError('option %s: invalid point value: %r' % (opt, value))


def check_settlement(option, opt, value):
    point = check_point(option, opt, value)
    if isinstance(point, Settlement):
        return point
    if isinstance(point, Station):
        if point.settlement:
            return point.settlement
    raise OptionValueError('option %s: invalid settlement value: %r' % (opt, value))


def check_query_key(option, opt, value):
    try:
        qkey, service = value.split('.')
        return Query.from_key(qkey, service=service)
    except Exception:
        pass

    raise OptionValueError(
        'option %s: invalid key: %r' % (opt, value))


def check_partner(option, opt, value):
    if value.startswith('dohop_'):
        dohop_id = int(value[len('dohop_'):])
        try:
            return DohopVendor.objects.get(dohop_id=dohop_id)
        except DohopVendor.DoesNotExist:
            pass

    if value.startswith('amadeus_'):
        try:
            return AmadeusMerchant.objects.get(code=value)
        except AmadeusMerchant.DoesNotExist:
            pass

    try:
        return Partner.objects.get(code=value)
    except Partner.DoesNotExist:
        pass

    raise OptionValueError(
        'option %s: partner_code %r not in list: %s' %
        (opt, value, ', '.join(sorted(partner_codes()))))


class Yoption(Option):
    TYPES = Option.TYPES + ('date', 'point', 'query_key', 'partner', 'settlement')
    TYPE_CHECKER = copy(Option.TYPE_CHECKER)
    TYPE_CHECKER['date'] = check_date
    TYPE_CHECKER['point'] = check_point
    TYPE_CHECKER['settlement'] = check_settlement
    TYPE_CHECKER['query_key'] = check_query_key
    TYPE_CHECKER['partner'] = check_partner


def main():
    parser = OptionParser(option_class=Yoption)
    parser.add_option('-k', '--key', dest='query', type='query_key')
    parser.add_option('-f', '--point_from', dest='point_from', type='point')
    parser.add_option('-t', '--point_to', dest='point_to', type='point')
    parser.add_option('-d', '--dateForward', dest='date_forward', type='date', help='date in format 24.05.2012')
    parser.add_option('-b', '--dateBackward', dest='date_backward', type='date', help='date in format 24.05.2012')
    parser.add_option('-n', '--national', dest='national_version', type='string', help='ru|ua|com|tr')
    parser.add_option('-q', '--class', dest='service_class', type='string', default='economy', help='economy | first | business')
    parser.add_option('-a', '--adults', dest='adults', type='int', default=1)
    parser.add_option('-c', '--children', dest='children', type='int', default=0)
    parser.add_option('-i', '--infants', dest='infants', type='int', default=0)
    parser.add_option('-p', '--partner', dest='partner', type='partner')
    parser.add_option('-s', '--service', dest='service', default='ticket')
    parser.add_option('-l', '--lang', dest='lang', default='ru')

    parser.add_option('-R', '--show_results', dest='show_results', action='store_true')
    parser.add_option('-T', '--trace', dest='trace', action='store_true')
    parser.add_option('-V', '--verbose', dest='verbose', action='store_true')

    (options, args) = parser.parse_args()

    if options.verbose:
        log = logging.getLogger('ticket_daemon')
        log.addHandler(logging.StreamHandler(sys.stdout))
        log.setLevel(logging.DEBUG)

    run(options, args)
