# -*- coding: utf-8 -*-
import logging
import multiprocessing
import os
import socket

from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.mdb.mysql import get_replica_info
from travel.avia.library.python.common.settings.migration import *  # noqa
from travel.avia.library.python.environment_helpers import parse_env_list
from travel.avia.library.python.redis.settings import RedisConfigurator

from travel.avia.ticket_daemon_api.environment import YANDEX_ENVIRONMENT_TYPE, env_variable_provider
from travel.avia.ticket_daemon_api.jsonrpc.lib.log_setup import create_logging_setting
from travel.avia.ticket_daemon_api.jsonrpc.ydb_settings import (  # noqa
    DRIVER_CONFIG,
    WIZARD_DRIVER_CONFIG,
    SESSION_POOL_SIZE,
    RECREATE_YDB_SESSION_POOL,
)

# system
ENVIRONMENT = {
    'production': 'production',
    'testing': 'testing',
    'development': 'dev',
    'load': 'testing',
}[YANDEX_ENVIRONMENT_TYPE]
CURRENT_DC = os.getenv('DEPLOY_NODE_DC')

PKG_VERSION = os.getenv('DEPLOY_DOCKER_IMAGE') or 'dev-{}'.format(os.getenv('USER'))
PROJECT_PATH = os.path.abspath(os.path.dirname(__file__))

LOG_PATH = env_variable_provider.get('log_path')
YT_LOG_PATH = env_variable_provider.get('YT_LOG_PATH')

LOGGING = create_logging_setting(LOG_PATH, YT_LOG_PATH, YANDEX_ENVIRONMENT_TYPE)
LOG_LEVEL = logging.INFO

DEBUG = False
DEFAULT_FILE_STORAGE = 'travel.avia.library.python.common.utils.fields.TransliteratingFileSystemStorage'
DEFAULT_FROM_EMAIL = 'rasp-tech@yandex-team.ru'
LANGUAGE_CODE = 'ru'
LANGUAGE_COOKIE_NAME = 'lang'
ROOT_URLCONF = 'urls'
SECRET_KEY = env_variable_provider.get(
    'SECRET_KEY'
)
TIME_ZONE = 'Europe/Moscow'
USE_TZ = True
USE_L10N = True
SERVER_EMAIL = 'Rasp at %s <rasp-tech@yandex-team.ru>' % socket.gethostname()
LOCALE_PATHS = [
    os.path.join(
        os.path.dirname(__file__),
        '..', 'virtualenv', 'django', 'conf', 'locale'
    )
]
EMAIL_SUBJECT_PREFIX = '[Django-avia@{}] '.format(socket.gethostname())
STATIC_URL = u''
INSTALLED_APPS = (
    'django.contrib.auth',
    'django.contrib.contenttypes',
    # Uncomment the next line to enable the admin:
    # 'django.contrib.admin',
    # Uncomment the next line to enable admin documentation:
    # 'django.contrib.admindocs',
    'travel.avia.library.python.common',
    'travel.avia.library.python.avia_data',
    'travel.avia.library.python.common.app_stubs.www',
    'travel.avia.library.python.common.app_stubs.currency',
    'travel.avia.library.python.common.app_stubs.order',
)

# caches
CACHE_REDIS = bool(env_variable_provider.get('CACHE_REDIS', required=False) == 'true')
SHARED_CACHE_REDIS = bool(env_variable_provider.get('SHARED_CACHE_REDIS', required=False) == 'true')
REDIS_PING_SLEEP_TIME = int(env_variable_provider.get('REDIS_PING_SLEEP_TIME', required=False, default=1000))
REDIS_MAXIMUM_LAG = int(env_variable_provider.get('MAXIMUM_REDIS_REPLICA_LAG', required=False, default=3000))

MDB_API_TOKEN = env_variable_provider.get('MDB_API_TOKEN', required=CACHE_REDIS)


def mcr_caches(hosts):
    if not hosts:
        raise ValueError('cannot configure mcrouter cache without hosts')

    return {
        'default': {
            'LOCATION': hosts,
            'TIMEOUT': 60,
            'LONG_TIMEOUT': 24 * 60 * 60,
            'OPTIONS': {'server_max_value_length': 128 * 1024 * 1024},
            'BACKEND': 'travel.avia.ticket_daemon_api.jsonrpc.lib.caches.LoggingMemcachedCache'
        }
    }


MCR_DAEMON_CACHES = mcr_caches(env_variable_provider.get('MCR_HOST', required=False, default='localhost:11211'))


_metric_queue = None


def get_metric_queue():
    global _metric_queue
    if _metric_queue is None:
        _metric_queue = multiprocessing.Queue()
    return _metric_queue


def default_cache():
    if CACHE_REDIS:
        return RedisConfigurator(
            service_name=env_variable_provider.get('REDIS_SENTINEL_SERVICE_NAME'),
            redis_cluster_id=env_variable_provider.get('REDIS_CLUSTER_ID'),
            redis_password=env_variable_provider.get('REDIS_PASSWORD'),
            redis_port=int(env_variable_provider.get('REDIS_PORT', required=False, default=26379)),
            socket_timeout=float(env_variable_provider.get('REDIS_SOCKET_TIMEOUT', required=False, default=0.1)),
            oauth_token=MDB_API_TOKEN,
            monitoring_sensor_prefix='default_cache',
            monitoring_metric_queue=get_metric_queue(),
        ).configure_redis()['default']
    else:
        return MCR_DAEMON_CACHES['default']


def shared_cache():
    if SHARED_CACHE_REDIS:
        return RedisConfigurator(
            service_name=env_variable_provider.get('SHARED_REDIS_SENTINEL_SERVICE_NAME'),
            redis_cluster_id=env_variable_provider.get('SHARED_REDIS_CLUSTER_ID'),
            redis_password=env_variable_provider.get('SHARED_REDIS_PASSWORD'),
            redis_port=int(env_variable_provider.get('SHARED_REDIS_PORT', required=False, default=26379)),
            socket_timeout=float(env_variable_provider.get('SHARED_REDIS_SOCKET_TIMEOUT', required=False, default=0.1)),
            oauth_token=MDB_API_TOKEN,
            monitoring_sensor_prefix='shared_cache',
            monitoring_metric_queue=get_metric_queue(),
        ).configure_redis_master_only()['default']
    else:
        return MCR_DAEMON_CACHES['default']


CACHES = {
    'default': default_cache(),
    'shared_cache': shared_cache(),
    'replica_set': {
        'BACKEND': 'django.core.cache.backends.locmem.LocMemCache',
        'timeout': 60
    }
}

CACHE_PATH = '/var/cache/yandex-ticket-daemon-api'

CACHEROOT = env_variable_provider.get('CACHEROOT', required=False, default='/{}/'.format(PKG_VERSION))
TICKET_DAEMON_CACHEROOT = (
    env_variable_provider.get('TICKET_DAEMON_CACHEROOT', required=False) or
    '/yandex/ticket-daemon/'
)

# national version and langs
ALLOW_LANGS = ['ru', 'uk', 'en', 'tr']
MODEL_LANGUAGES = ['ru', 'en', 'tr', 'uk']
LANGUAGE_CASE_FALLBACKS = {('uk', 'genitive'): [('ru', 'genitive')]}
LANGUAGE_FALLBACKS = {'de': ['en'], 'tr': ['en'], 'uk': ['ru']}
TRANSLATIONS_ALLOW_LANGS = ['ru', 'uk', 'en', 'tr', 'de']

AVIA_NATIONAL_VERSIONS = ['ru', 'ua', 'tr', 'com', 'kz']
DEFAULT_NATIONAL_VERSION_LANGUAGES = {
    'ru': 'ru',
    'kz': 'ru',
    'ua': 'uk',
    'tr': 'tr',
    'com': 'en',
}
AVIA_NATIONAL_VERSIONS_CHOICES = tuple([
    (v, v) for v in AVIA_NATIONAL_VERSIONS]
)
AVIA_NATIONAL_VERSIONS_CHOICES_MAX_LEN = 5
ALLOW_NATIONAL_VERSIONS = ['ru', 'com', 'tr', 'ua', 'kz']
TRAVEL_HOST_BY_NATIONAL_VERSION = {
    'ru': env_variable_provider.get('TRAVEL_HOST_FOR_RU'),
}
AVIA_HOST_BY_NATIONAL_VERSION = {
    'ru': env_variable_provider.get('FRONT_HOST_FOR_RU'),
    'kz': env_variable_provider.get('FRONT_HOST_FOR_KZ'),
    'ua': env_variable_provider.get('FRONT_HOST_FOR_UA'),
    'tr': env_variable_provider.get('FRONT_HOST_FOR_TR'),
    'com': env_variable_provider.get('FRONT_HOST_FOR_COM'),
}
LANGUAGES = [
    ('ru', _(u'Русский')),
    ('be', _(u'Белорусский')),
    ('kk', _(u'Казахский')),
    ('tt', _(u'Татарский')),
    ('uk', _(u'Украинский')),
    ('en', _(u'Английский')),
    ('tr', _(u'Турецкий'))
]

# currencies
AVIA_NATIONAL_CURRENCIES = {
    'ru': 'RUR',
    'ua': 'UAH',
    'kz': 'KZT',
    'tr': 'TRY',
    'com': 'EUR',
}
NATIONAL_CURRENCIES = {'ru': 'RUR', 'tr': 'TRY', 'ua': 'UAH'}
NATIONAL_CURRENCY_RATES_GEOID = {
    'com': 10376, 'kz': 163, 'ru': 213, 'tr': 11508, 'ua': 143
}
CURRENCY_RATES_TIMEOUT = 0.1
CURRENCY_RATES_URL = env_variable_provider.get(
    'CURRENCY_RATES_URL'
)

MYSQL_CLUSTER_ID = env_variable_provider.get('MYSQL_CLUSTER_ID', required=False)

MYSQL_CONNECT_TIMEOUT = 1
MYSQL_READ_TIMEOUT = 60
MYSQL_WRITE_TIMEOUT = 3

DATABASES = {
    'default': {
        'ATOMIC_REQUESTS': False,
        'AUTOCOMMIT': True,
        'CONN_MAX_AGE': 0,
        'ENGINE': 'travel.avia.ticket_daemon_api.jsonrpc.lib.mysql_switcher',
        'HOST': None,
        'NAME': env_variable_provider.get('MYSQL_DATABASE_NAME'),
        'OPTIONS': {},
        'PASSWORD': env_variable_provider.get('MYSQL_PASSWORD', required=False),
        'PORT': env_variable_provider.get('MYSQL_PORT', required=False) or '',
        'STORAGE_ENGINE': 'InnoDB',
        'TEST': {
            'CHARSET': None,
            'COLLATION': None,
            'MIRROR': None,
            'NAME': None
        },
        'TIME_ZONE': 'Europe/Moscow',
        'USER': env_variable_provider.get('MYSQL_USER')
    }
}

if MYSQL_CLUSTER_ID:
    replica_info = get_replica_info({
        'MYSQL_CLUSTER_ID': MYSQL_CLUSTER_ID,
        'MDB_API_TOKEN': MDB_API_TOKEN,
    }, CURRENT_DC)
    DATABASES['default']['REPLICAS'] = replica_info.replicas
else:
    DATABASES['default']['HOST'] = env_variable_provider.get('MYSQL_HOST', required=False) or 'localhost'

TEST_DB_NAME = env_variable_provider.get(
    'MYSQL_TEST_DATABASE_NAME',
    required=False
)

# app
ALLOW_GEOBASE = False
TICKET_DAEMON_URL = env_variable_provider.get('TICKET_DAEMON_URL')
PARTNER_QUERY_TIMEOUT = int(env_variable_provider.get(
    'PARTNER_QUERY_TIMEOUT'
))
DAEMON_API_HTTP_TIMEOUT = 5.0
DAEMON_API_REDIRECT_TIMEOUT = 9.6
CHARTER_PARTNERS = ['charterbilet', 'aviaoperator']
NEMO_PARTNERS = frozenset({'agent', 'biletik', 'expressavia', 'rusline', 'tez_tour', 'uzairways', 'biletinet'})

GUNICORN_PIDFILE = '/run/yandex-ticket-daemon/gunicorn_ticket_daemon.pid'
HTTP_TIMEOUT = 3
QUICK_HTTP_TIMEOUT = 0.1

FEATURE_FLAG_HOST = env_variable_provider.get(
    'FEATURE_FLAG_HOST'
)

SENTRY_DSN = env_variable_provider.get(
    'SENTRY_DSN'
)

BOOK_ON_YANDEX_REQUEST_TIMEOUT = float(env_variable_provider.get(
    'BOOK_ON_YANDEX_REQUEST_TIMEOUT',
    required=False,
) or 9)

AFL_PARTNER_CODE = 'aeroflot'

BOOKING_TESTERS = {
    user_login.strip()
    for user_login in (env_variable_provider.get(
        'BOOKING_TESTERS', required=False) or 'yndx-tcheremnykh, tlg-13, slow.p0ke, gouken457, backorange').split(',')
}

BOOKING_TESTERS_AEROFLOT = {
    user_login.strip()
    for user_login in (env_variable_provider.get(
        'BOOKING_TESTERS_AEROFLOT', required=False) or '').split(',')
}

BOY_DISABLED_DEPARTURE_AIRPORT_CODES = env_variable_provider.get(
    'BOY_DISABLED_DEPARTURE_AIRPORT_CODES',
    required=False,
    default='',
).split(',')

ENABLE_TVM = env_variable_provider.get('ENABLE_TVM', required=False) == 'true'
TVM_TIMEOUT = float(env_variable_provider.get('TVM_TIMEOUT', required=False, default=0.01))
if ENABLE_TVM:
    TVM_TOKEN = os.environ['TVMTOOL_LOCAL_AUTHTOKEN']
    TVM_NAME = env_variable_provider.get('TVM_NAME')
    TVM_DESTINATIONS = [
        int(d) for d in env_variable_provider.get(
            'TVM_DESTINATIONS'
        ).split(',')
    ]

STATSD_HOST = 'localhost'
STATSD_PORT = 8125
STATSD_PREFIX = None
STATSD_MAXUDPSIZE = 512

DEVELOPMENT_FLAGS = os.getenv('DEVELOPMENT_FLAGS', '').split(',')

COVID19_ZERO_REDIRECT_PRICE = os.getenv('AVIA_COVID19_ZERO_REDIRECT_PRICE', default=False)
COVID19_ZERO_REDIRECT_PRICE_NATIONAL_VERSIONS = {'ru'}
COVID19_ZERO_REDIRECT_PRICE_PARTNERS = frozenset([
    'agent',
    'anywayanyday',
    'aviakass',
    'aviaoperator',
    'biletdv',
    'biletikaeroag',
    'biletinet',
    'biletix',
    'booktripruag',
    'citytravel',
    'citytravel1',
    'clickaviatravelab',
    'expressavia',
    'kiwi',
    'kupibilet',
    'megotravel',
    'nebotravel',
    'ozon',
    'superkassa',
    'supersaver',
    'svyaznoy',
    'ticketsru',
    'tripcom',
    'trip_ru',
    'uzairways',
])

BANNED_PARTNERS_FOR_RESULTS = set(
    parse_env_list(
        env_variable_provider.get(
            'BANNED_PARTNERS_FOR_RESULTS',
            required=False,
            default='tutu',
        ),
    ),
)

TEMPLATES = [{}]  # workaround for patch_settings_for_arcadia

TRAVEL_APP_SERVICE_NAME = 'travelapp'

from library.python.django.utils import patch_settings_for_arcadia  # noqa

patch_settings_for_arcadia()
