# -*- coding: utf-8 -*-
import logging
import os
import time

import django
django.setup()
from django.conf import settings
from flask import Flask, send_from_directory, request
from flask_swagger_ui import get_swaggerui_blueprint

from travel.avia.library.python.flask_helpers import wsgi_middleware
from travel.avia.ticket_daemon_api.jsonrpc.handlers.avia_api.views import avia_api
from travel.avia.ticket_daemon_api.jsonrpc.handlers.rasp.views import rasp_api
from travel.avia.ticket_daemon_api.jsonrpc.handlers.td_processing.views import td_processing_api
from travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.views import api as api3
from travel.avia.ticket_daemon_api.jsonrpc.handlers.wizard.views import wizard_api
from travel.avia.ticket_daemon_api.jsonrpc.health import available_ping_checks, HealthError
from travel.avia.ticket_daemon_api.jsonrpc.views import api, front_api


log = logging.getLogger(__name__)

if settings.DEBUG:
    import sys
    formatter = logging.Formatter(
        '(%(asctime)s) %(module)s:%(funcName)s [%(levelname)s] %(message)s')
    logging.getLogger().setLevel(settings.LOG_LEVEL)
    h = logging.StreamHandler(sys.stdout)
    logging.getLogger().addHandler(h)
    for h in logging.getLogger().handlers:
        h.setLevel(settings.LOG_LEVEL)
        h.setFormatter(formatter)


def register_swagger(app):
    swagger_url = '/api/docs'
    swaggerui_blueprint = get_swaggerui_blueprint(
        swagger_url,
        '/api/docs/swagger.json',
        config={
            'app_name': 'Ticket Daemon API'
        },
    )
    app.register_blueprint(swaggerui_blueprint, url_prefix=swagger_url)

    @app.route('/api/docs/swagger.json', methods=['GET'])
    def swagger():
        return send_from_directory(
            os.path.join(settings.PROJECT_PATH, 'jsonrpc'), 'swagger.json'
        )


class App(Flask):
    def __init__(self, *args, **kwargs):
        super(App, self).__init__(*args, **kwargs)
        self.shutdown_flag = None
        self.cache_readiness_flag = None
        self.shared_cache_readiness_flag = None


def create_app(config):
    app = App(__name__)
    app.config['SECRET_KEY'] = 'open'
    if settings.DEBUG:
        app.debug = True

    app.config.update(config)
    app.url_map.strict_slashes = False

    app.register_blueprint(api, url_prefix='/jsendapi')
    app.register_blueprint(front_api, url_prefix='/jsendapi/front')
    app.register_blueprint(rasp_api, url_prefix='/jsendapi/rasp')
    app.register_blueprint(avia_api, url_prefix='/jsendapi/avia_api')
    app.register_blueprint(api3, url_prefix='/jsendapi/v3')
    app.register_blueprint(wizard_api, url_prefix='/jsendapi/wizard/v1')
    app.register_blueprint(td_processing_api, url_prefix='/jsendapi/td_processing/v1')
    register_swagger(app)

    app.wsgi_app = wsgi_middleware.HTTPSchemeProxy(app.wsgi_app)
    app.wsgi_app = wsgi_middleware.Now(app.wsgi_app)
    app.wsgi_app = wsgi_middleware.ABExperimentFlags(app.wsgi_app)

    @app.route('/ping')
    def ping():
        if app.shutdown_flag.is_set():
            return 'stopped', 410

        try:
            if not app.cache_readiness_flag.is_set():
                return 'cache_readiness_flag is not set', 503
            if not app.shared_cache_readiness_flag.is_set():
                return 'shared_cache_readiness_flag is not set', 503
            for check in available_ping_checks:
                if request.args.get(check.disable_flag) is None:
                    check.checker.check()
        except HealthError as e:
            log.warn(e, exc_info=True)
            return 'error', 503
        except Exception:
            log.exception('ping td-api exception')
            return 'error', 500

        return 'alive', 200

    @app.route('/shutdown', methods=['POST'])
    def shutdown():
        max_delay = 20  # seconds

        delay = int(request.args.get('delay', max_delay))
        if delay > max_delay:
            return 'max delay: {}'.format(max_delay), 400

        try:
            app.shutdown_flag.set()
            time.sleep(delay)
        except Exception:
            log.exception('shutdown exception')
            return 'error', 500
        else:
            return 'ok', 200

    return app
