# -*- coding: utf-8 -*-
import gc
import os
import socket

import django

from travel.library.python.gunicorn.worker_watcher import WorkersWatcher
from travel.library.python.solomon.server import SolomonMetricsServer

from travel.avia.library.python.redis.ping import RedisBackgroundPing
from travel.avia.library.python.shared_objects import SharedFlag
from travel.avia.library.python.ticket_daemon.memo import reset_all_caches

from travel.avia.ticket_daemon_api.api_settings import (
    CACHEROOT,
    CACHE_REDIS,
    REDIS_MAXIMUM_LAG,
    REDIS_PING_SLEEP_TIME,
    SHARED_CACHE_REDIS,
    get_metric_queue,
)
from travel.avia.ticket_daemon_api.environment import env_variable_provider
from travel.avia.ticket_daemon_api.jsonrpc.lib.caches import default_cache, shared_cache

name = 'yandex-ticket-daemon-api'

# Тут нужно аккуратно, права на запись затираются после рестарта машины
# >> because /run/ in mounted as tmpfs and cleared after each server reboot.
# В нашем случае каталог с правами создаётся из скрипта запуска самого тд
bind = env_variable_provider.get('bind_address')

loglevel = env_variable_provider.get('log_level')
errorlog = env_variable_provider.get('gunicorn_log')
capture_output = True

worker_class = 'travel.library.python.gunicorn.worker_watcher.workers.SyncWorker'

workers = int(env_variable_provider.get('gunicorn_workers_count'))
worker_connections = 1000
max_requests = int(env_variable_provider.get('gunicorn_max_requests', required=False) or 100000)
max_requests_jitter = max_requests // 5

VERSION = os.getenv('DEPLOY_DOCKER_IMAGE', ':unknown').rsplit(':', 1)[-1]
TRACING_SERVICE_NAME = 'avia-ticket-daemon-api'
TRACING_JAEGER_SAMPLER_TIPE = os.getenv('JAEGER_SAMPLER_TYPE', 'probabilistic')
TRACING_JAEGER_SAMPLER_PARAMETER = float(os.getenv('JAEGER_SAMPLER_PARAMETER', 0.001))

shutdown_flag = SharedFlag()
cache_readiness_flag = SharedFlag()
shared_cache_readiness_flag = SharedFlag()

solomon_server = SolomonMetricsServer()
solomon_server.register_metric_queue(get_metric_queue())

key_ttl = 60
if CACHE_REDIS:
    cache_ping = RedisBackgroundPing(
        default_cache,
        '{}ping_check/{}'.format(CACHEROOT, socket.gethostname()),
        key_ttl,
        REDIS_PING_SLEEP_TIME,
        REDIS_MAXIMUM_LAG,
        cache_readiness_flag,
    )
if SHARED_CACHE_REDIS:
    shared_cache_ping = RedisBackgroundPing(
        shared_cache,
        '{}shared_ping_check/{}'.format(CACHEROOT, socket.gethostname()),
        key_ttl,
        REDIS_PING_SLEEP_TIME,
        0,
        shared_cache_readiness_flag,
    )

watcher = WorkersWatcher()


def precache(server):
    from travel.avia.ticket_daemon_api.jsonrpc.models_utils import warm_up_cache

    try:
        server.log.info('Warming up...')
        warm_up_cache(logger=server.log)
    except Exception as e:
        server.log.exception('Fail warming up%r', repr(e))
        raise e


def on_starting(server):
    django.setup()
    server.log.info('start master process...')

    watcher.on_starting(server)

    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work'}, watcher.get_requests_in_work_count)
    solomon_server.register_gauge({'sensor': 'gunicorn.requests_in_work_max'}, watcher.get_max_requests_in_work)
    solomon_server.run_in_thread()

    if CACHE_REDIS:
        cache_ping.run()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.run()
    precache(server)
    server.log.info('... finish start master process')


def on_reload(server):
    server.log.info('Update cache ...')
    if CACHE_REDIS:
        cache_ping.run()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.run()
    reset_all_caches()
    precache(server)
    server.log.info('... finish update cache')


def on_exit(server):
    if CACHE_REDIS:
        cache_ping.stop()
    if SHARED_CACHE_REDIS:
        shared_cache_ping.stop()
    solomon_server.stop_thread()


def pre_fork(server, worker):
    """
    Закрываем соединение, чтобы не передавать воркерам
    копию сокета [RASPWIZARDS-175]
    """
    from django.core.cache import cache
    cache.close()

    watcher.pre_fork(server, worker)

    # собираем мусор, чтобы не оставлять его в форках
    gc.collect()


def child_exit(server, worker):
    watcher.child_exit(server, worker)


def post_fork(server, worker):
    from flask_opentracing_helpers import setup_tracing

    from travel.avia.ticket_daemon_api.api_wsgi import application

    application.shutdown_flag = shutdown_flag
    application.cache_readiness_flag = cache_readiness_flag
    application.shared_cache_readiness_flag = shared_cache_readiness_flag
    if not CACHE_REDIS:
        application.cache_readiness_flag.set()
    if not SHARED_CACHE_REDIS:
        application.shared_cache_readiness_flag.set()

    server.log.info('Setup tracing...')
    env_variables = ['DEPLOY_STAGE_ID', 'DEPLOY_NODE_DC', 'DEPLOY_BOX_ID']
    try:
        setup_tracing(
            flask_application=application,
            service_name=TRACING_SERVICE_NAME,
            version=VERSION,
            include_env_variables=env_variables,
            filter_paths=['api.ping'],
            jaeger_sampler_type=TRACING_JAEGER_SAMPLER_TIPE,
            jaeger_sampler_parameter=TRACING_JAEGER_SAMPLER_PARAMETER
        )
    except Exception:
        server.log.exception('Failed to setup tracing')
    else:
        server.log.info('Done setup tracing successfully')
