# -*- coding: utf-8 -*-
import logging
from datetime import datetime
from itertools import chain

from django.conf import settings

from travel.avia.ticket_daemon_api.jsonrpc.complete_results import avia_deep_link, avia_order_link
from travel.avia.ticket_daemon_api.jsonrpc.lib.currencies import CurrencyConverter

log = logging.getLogger(__name__)

DT_FORMAT = '%Y-%m-%dT%H:%M:%S'
OLD_KEY_DT_FORMAT = '%m%dT%H%M'


def _key1(routes, flights):
    return u'_'.join(
        u'{}.{}'.format(
            flights[route]['number'],
            datetime.strptime(flights[route]['departure']['local'], DT_FORMAT).strftime(OLD_KEY_DT_FORMAT) if flights[route]['departure'].get('local') else ''
        ) for route in routes
    )


def _key2(routes, flights):
    return u','.join(
        u'{}.{}'.format(
            flights[route]['number'],
            flights[route]['departure'].get('local', '')[:-3]
        ) for route in routes
    )


def _flight_key(f):
    return '-'.join(map(str, (
        f['number'].encode('utf-8'),
        f['company'],
        f['from'],
        f['to'],
        datetime.strptime(f['departure']['local'], DT_FORMAT).strftime(OLD_KEY_DT_FORMAT) if f['departure'] and f['departure'].get('local') else '',
        datetime.strptime(f['arrival']['local'], DT_FORMAT).strftime(OLD_KEY_DT_FORMAT) if f['arrival'] and f['arrival'].get('local') else '',
    )))


def complete_results_for_avia_api(
    query, results, redirect_datum, status, currency, clid
):
    """

    :type query: jsonrpc.query.Query
    :type results: typing.Dict[str, jsonrpc.lib.result.collector.variants_fabric.ApiVariants]
    :param redirect_datum:
    :param status:
    :param currency:
    :param clid:
    :return:
    """
    rates = CurrencyConverter(query.national_version)

    trips = {}
    flights = {}

    full_variants = []
    not_full_variants = []
    for p_code, variants in results.iteritems():
        v_flights = dict(variants.flights)
        completed_flight_keys = {
            f_key for f_key, f in v_flights.iteritems()
            if f['number'] and f['departure'] and f['arrival']
        }
        for v in variants.variants:
            if not all(
                route in completed_flight_keys
                for route in chain.from_iterable(v['route'])
            ):
                not_full_variants.append(v)
                continue

            # params for avia_deeplink
            v['partner_code'] = p_code
            v['forward'] = {
                'key2': _key2(v['route'][0], v_flights),
            }
            v['backward'] = {
                'key2': _key2(v['route'][1], v_flights),
            } if v['route'][1] else None

            fwd_key = _key1(v['route'][0], v_flights)
            bwd_key = _key1(v['route'][1], v_flights) if v['route'][1] else ''
            result_variant = {
                'forward': fwd_key,
                'backward': bwd_key,
                'partner': p_code,
                'base_tariff': {
                    'currency': v['tariff']['currency'],
                    'value': v['tariff']['value'],
                },
                'tariff': {
                    'currency': currency,
                    'value': rates.convert_value(
                        value=v['tariff']['value'],
                        from_currency=v['tariff']['currency'],
                        to_currency=currency,
                    ),
                },
                'tag': v['tag'],
                'charter': p_code in settings.CHARTER_PARTNERS,
                'query_time': variants.query_time,
                'order_data': dict(
                    partner=p_code,
                    qkey=query.qkey,
                    tariff=v['tariff'],
                ),
                'expires_at': {
                    'tz': 'UTC',
                    'local': datetime.utcfromtimestamp(v['expire']).strftime(DT_FORMAT),
                },
                'deep_link': avia_deep_link(query, v, clid),
                'order_link': avia_order_link(query, v, clid),
            }

            full_variants.append(result_variant)

            for routes, trip_key in zip(v['route'], [fwd_key, bwd_key]):
                if trip_key not in trips:
                    for route in routes:
                        seg_key = _flight_key(v_flights[route])
                        trips.setdefault(trip_key, []).append(seg_key)

                        if seg_key not in flights:
                            flights[seg_key] = v_flights[route]

    if not_full_variants:
        log.info(
            '[%s] Discarded %d/%d variants', query.id, len(not_full_variants),
            len(not_full_variants) + len(full_variants)
        )
        if (
            not full_variants
            or (
                min(v['tariff']['value'] for v in full_variants) >
                min(v['tariff']['value'] for v in not_full_variants)
            )
        ):
            log.info('[%s] Lost min price!', query.id)

    def _as_string(dt_dict):
        return {
            'tz': dt_dict['tzname'],
            'local': dt_dict['local']
        }

    return {
        'status': status,
        'variants': full_variants,
        'trips': trips,
        'flights': {
            key: {
                'number': flight['number'],
                'company': flight['company'],
                'from': flight['from'],
                'to': flight['to'],
                'departs_at': _as_string(flight['departure']),
                'arrives_at': _as_string(flight['arrival']),
            } for key, flight in flights.iteritems()
        },
    }
