# -*- coding: utf-8 -*-
import logging
import json
from typing import Any

from flask import Blueprint, request
from werkzeug.exceptions import BadRequest

from travel.avia.library.python.ticket_daemon.query_validation.validation import validate_query, ValidationError
from travel.avia.ticket_daemon_api.environment import YANDEX_ENVIRONMENT_TYPE
from travel.avia.ticket_daemon_api.jsonrpc.handlers.avia_api.complete_results import complete_results_for_avia_api
from travel.avia.ticket_daemon_api.jsonrpc.query import Query, init_search
from travel.avia.ticket_daemon_api.jsonrpc.handlers.avia_api.scheme import AviaApiSearchResultsInputSchema
from travel.avia.ticket_daemon_api.jsonrpc.handlers.scheme import QueryParamsSchema, RedirectDataInputSchema
from travel.avia.ticket_daemon_api.jsonrpc.lib import result as resultlib
from travel.avia.ticket_daemon_api.jsonrpc.lib.internal_daemon_client import internal_daemon_client
from travel.avia.ticket_daemon_api.jsonrpc.lib.partner_timeout_provider import partner_redirect_timeout_provider
from travel.avia.ticket_daemon_api.jsonrpc.lib.redirect import url_add_openstat_marker
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.test_context import parse_test_context_proto
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import skip_None_values
from travel.avia.ticket_daemon_api.jsonrpc.lib.yaclid import YaClid
from travel.avia.ticket_daemon_api.jsonrpc.views import jsend_view, clean_query_from_qid, api


log = logging.getLogger(__name__)

avia_api = Blueprint('avia_api', __name__)


def _make_search_query(data):
    # type: (Any) -> Query
    query_params, errors = QueryParamsSchema().load(data)
    if errors:
        log.warning('Query params are not valid: %r', errors)
        raise BadRequest(repr(errors))

    query = Query(**query_params)
    try:
        validate_query(query)
    except ValidationError as e:
        raise BadRequest(e.data)
    return query


@avia_api.route('/init_search/')
@jsend_view
def avia_api_init_search():
    query = _make_search_query(request.values)

    # https://st.yandex-team.ru/RASPTICKETS-9673
    if query.service == 'tours':
        return {'qid': query.id}

    # https://st.yandex-team.ru/RASPTICKETS-21483
    if query.test_context and YANDEX_ENVIRONMENT_TYPE != 'production':
        tk = parse_test_context_proto(query.test_context)
        if tk.MockAviaVariants:
            return {'qid': query.id}

    return init_search(query)


@avia_api.route('/results/')
@jsend_view
def avia_api_results():
    params, errors = AviaApiSearchResultsInputSchema().load(request.values)
    if errors:
        log.warning('Api search results. Params are not valid: %r', errors)
        raise BadRequest(repr(errors))

    q = clean_query_from_qid(request.values.get('qid'))
    # https://st.yandex-team.ru/RASPTICKETS-9673
    # https://st.yandex-team.ru/RASPTICKETS-18678
    if q.service != 'sovetnik':
        return {
            'status': {},
            'variants': [],
            'trips': {},
            'flights': {},
        }

    tags = set(params['tags'])

    redirect_datum = {}
    results, statuses, _ = resultlib.collect_variants(
        q,
        skip_partner_codes=params['skip'],
        mode=resultlib.CollectingModes.actual,
        prefilter=(lambda v: v['tag'] in tags) if tags else None,
    )

    return complete_results_for_avia_api(
        q, results, redirect_datum, statuses, params['currency'], params['clid']
    )


@avia_api.route('/redirect_data/')
@jsend_view
def avia_api_redirect_data():
    params, errors = RedirectDataInputSchema().load(request.values)
    if errors:
        log.warning('Redirect data. Params are not valid: %r', errors)
        raise BadRequest(repr(errors))

    order_data = json.loads(request.values.get('order_data'))
    order_data.update(params['order_data'])

    redirect_data = internal_daemon_client.redirect(
        order_data=order_data,
        user_info=params['user_info'],
        additional_data=params,
        utm_source='api_avia',
        query_source=order_data.get('query_source'),
        timeout=partner_redirect_timeout_provider.get(
            partner_code=order_data['partner']
        )
    )

    query = clean_query_from_qid(params['qid'])

    if query.is_mobile:
        redirect_url = redirect_data.m_url or redirect_data.url
    else:
        redirect_url = redirect_data.url

    if not redirect_url:
        raise BadRequest('No redirect data')

    return skip_None_values({
        'url': url_add_openstat_marker(
            url=redirect_url,
            partner_code=order_data['partner'],
            host=request.environ.get('HTTP_HOST'),
            domain=query.national_version,
        ),
        'post': redirect_data.post,
    })


@api.route('/decode_yaclid/')  # without prefix avia_api
@jsend_view
def decode_yaclid():
    """Используется в avia_api для проверки валидности yaclid"""
    yaclid = YaClid.clean(request.values.get('yaclid') or '')
    if not yaclid:
        raise BadRequest('Malformed yaclid')
    return {
        'age_seconds': int(yaclid.age().total_seconds()),
    }


@avia_api.route('/redirect_page_qid/')
@jsend_view
def front_get_qid():
    """Используется в avia_api при составлении линка на страницу редиректа"""
    query_params, errors = QueryParamsSchema().load(request.values)
    if errors:
        log.warning('Get QID. Query params are not valid: %r', errors)
        raise BadRequest(repr(errors))
    q = Query(**query_params)

    return q.id
