# -*- coding: utf-8 -*-
import logging

from flask import Blueprint, request
from werkzeug.exceptions import BadRequest

from travel.avia.ticket_daemon_api.jsonrpc.handlers.rasp.scheme import (
    QueryResultsInputSchema, ResultsInputSchema
)
from travel.avia.ticket_daemon_api.jsonrpc.lib import result as resultlib
from travel.avia.ticket_daemon_api.jsonrpc.lib.flights import IATAFlight
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.continuation import get_rasp_cont_details, create_next_cont
from travel.avia.ticket_daemon_api.jsonrpc.query import Query, init_search
from travel.avia.ticket_daemon_api.jsonrpc.rasp.complete_results import rasp_complete_results
from travel.avia.ticket_daemon_api.jsonrpc.views import jsend_view, clean_query_from_qid, api, make_query

log = logging.getLogger(__name__)

rasp_api = Blueprint('rasp', __name__)


@api.route('/init_search/')  # without prefix /rasp
@jsend_view
def search_init():
    log.debug('Search init. Got unvalidated OPTIONS: %r', request.values)
    q = make_query(request.values)

    return init_search(q)


@rasp_api.route('/results_by_qid/')
@jsend_view
def results_by_qid():
    params, errors = ResultsInputSchema().load(request.values)
    if errors:
        log.warning('Rasp results by qid. Params are not valid: %r', errors)
        raise BadRequest(repr(errors))
    q = clean_query_from_qid(params['qid'])
    return _results_for_rasp(q, params['resultpages'], params['skip_partners'], params['full_results'])


@rasp_api.route('/results_by_search_params/')
@jsend_view
def results_by_search_params():
    query_params, errors = QueryResultsInputSchema().load(request.values)
    skip_partners = query_params.pop('skip_partners')
    resultpages = query_params.pop('resultpages')
    full_results = query_params.pop('full_results')
    if errors:
        log.warning('Rasp results by search params. Query params are not valid: %r', errors)
        raise BadRequest(repr(errors))
    q = Query(**query_params)
    return _results_for_rasp(q, resultpages, skip_partners, full_results)


def _results_for_rasp(q, result_pages, skip_partners, full_results):
    if result_pages:
        log.warning('result_pages will be deprecated for rasp handler')

    results, statuses, revisions = resultlib.collect_variants(
        q,
        skip_partner_codes=skip_partners,
        mode=resultlib.CollectingModes.actual,
        prefilter=lambda v: full_results or (len(v['route'][0]) == 1 and not v['route'][1]),
        # result_revisions=cont_details.revisions, #  Todo: Could be allowed with result_pages
    )
    cont_details = get_rasp_cont_details(q.id)

    in_progress_partners = [
        p_code for p_code, status in statuses.iteritems()
        if resultlib.Statuses.in_progress(status)
    ]
    completed_partners = {
        p_code for p_code, status in statuses.iteritems()
        if not resultlib.Statuses.in_progress(status)
    }
    actual_variants_count = sum([
        len(result) for p_code, result in results.iteritems()
        if result.status != resultlib.Statuses.OUTDATED
    ])

    next_cont_details = create_next_cont(
        cont_details,
        skip_partner_codes=completed_partners,
        revisions=revisions,
        in_progress=in_progress_partners,
        variants_count=actual_variants_count,
        is_rasp_cont=True,
    )

    status_code = 200
    if in_progress_partners:
        status_code = 206
    elif next_cont_details.variants_count == 0:
        status_code = 204
    answer = rasp_complete_results(q, results)

    return _serialize_rasp_results(q, answer, statuses), status_code


def _serialize_rasp_results(q, answer, status):
    reference = answer['reference']
    expired_date = answer['expired_date']
    flight_url_utms = {
        'utm_source': 'rasp',
        'utm_medium': 'search_result',
        'utm_campaign': 'flight_link',
    }

    def serialize_datetime(datetime_with_locale):
        if not datetime_with_locale:
            return

        return {
            'local': datetime_with_locale['local'].strftime('%Y-%m-%d %H:%M:%S'),
            'tzname': datetime_with_locale['tzname'],
        }

    return {
        'reference': {
            'flights': [{
                'key': f['key'],
                'number': f['number'],
                't_type_code': f['t_type_code'],
                'supplier_code': f['supplier_code'],
                'company': f['company'],
                'station_from': f['station_from'],
                'station_to': f['station_to'],
                'departure': serialize_datetime(f['departure']),
                'arrival': serialize_datetime(f['arrival']),
                'url': IATAFlight.url(f['number'], f['departure'], q.national_version, flight_url_utms),
            } for f in reference['flights']],
            'itineraries': reference['itineraries'],
            'settlements': [s.id for s in reference['settlements']],
            'stations': [s.id for s in reference['stations']],
            'partners': [
                {
                    'code': p.code,
                    'title': p.get_title(lang=q.lang, national_version=q.national_version),
                    'logoSvg': p.get_logo(national_version=q.national_version),
                } for p in reference['partners']
            ]
        },
        'variants': [{
            'forward': v['forward_itinerary'],
            'backward': v['backward_itinerary'],
            'order_data': {},
            'partner': v['partner_code'],
            'tariff': v['tariff'],
            'deep_link': v['deep_link'],
            'order_link': v['order_link'],
            'raw_tariffs': {},
            'query_time': int(v['query_time']),
            'from_company': v['from_company']
        } for v in answer['variants']],
        'status': status,
        'expired_date': expired_date.strftime('%Y-%m-%dT%H:%M:%S+00:00') if expired_date else None,
    }
