# -*- coding: utf-8 -*-
import json
from datetime import datetime, timedelta

from django.conf import settings
from marshmallow import Schema, fields, post_load, validates_schema
from marshmallow.exceptions import ValidationError

from travel.avia.library.python.ticket_daemon.caches.services import get_service_by_code
from travel.avia.library.python.ticket_daemon.query_validation.validation import ErrorCodes
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo import (
    get_point_by_iata_or_sirena, get_point_tuple_by_key,
)
from travel.avia.ticket_daemon_api.jsonrpc.query import Query, QueryIsNotValid


def _get_point(value):
    point = get_point_tuple_by_key(value) or get_point_by_iata_or_sirena(value.upper())
    if point:
        return point
    raise ValidationError('Point not found by code')


def clean_tags(value):
    if not value:
        return {}

    try:
        tags = json.loads(value)
    except Exception as e:
        raise ValidationError(repr(e))

    if (
        not isinstance(tags, list)
        or any(not isinstance(t, basestring) for t in tags)
    ):
        raise ValidationError(
            'tags should be a json-encoded list of strings')

    return tags


def clean_resultpages(value):
    try:
        data = json.loads(value) if value else {}
        return {str(pcode): int(page) for pcode, page in data.iteritems()}
    except Exception as e:
        raise ValidationError('resultpages: %r' % e)


def clean_skippartners(value):
    if not value:
        return []
    try:
        skip_partners = json.loads(value)
    except Exception as e:
        raise ValidationError(repr(e))
    if not isinstance(skip_partners, list):
        raise ValidationError('skip_partners should be a list of strings')
    for pcode in skip_partners:
        if not isinstance(pcode, basestring):
            raise ValidationError('skip_partners should be a list of strings')
    return skip_partners


def validate_service(value):
    if not bool(get_service_by_code(value)):
        raise ValidationError('Service %s not in PROJECT_GROUPS' % value)


class Qid(fields.String):
    def _deserialize(self, value, attr, data):
        qid = super(Qid, self)._deserialize(value, attr, data)

        try:
            q = Query.from_qid(qid)
            if q.id != qid:
                raise ValidationError('Invalid qid')
        except QueryIsNotValid as e:
            raise ValidationError(e.message)
        except ValueError:
            raise ValidationError('Parse error')

        return qid


class QidInputSchema(Schema):
    qid = Qid(required=True)


class QueryParamsSchema(Schema):
    # откуда (pointkey либо iata либо geoid)
    point_from = fields.Function(deserialize=_get_point, required=True)
    # куда
    point_to = fields.Function(deserialize=_get_point, required=True)

    # дата вылета туда
    date_forward = fields.Date(required=True)
    # дата вылета обратно
    date_backward = fields.Date(required=False, missing=None)

    # набор пассажиров
    adults = fields.Integer(missing=1, validate=fields.validate.Range(min=1, max=9))
    children = fields.Integer(missing=0, validate=fields.validate.Range(min=0, max=9))
    infants = fields.Integer(missing=0, validate=fields.validate.Range(min=0, max=9))

    # класс перелёта
    klass = fields.String(
        missing='economy',
        validate=fields.validate.OneOf(['economy', 'business'])
    )
    # язык
    lang = fields.String(
        required=True,
        validate=fields.validate.OneOf(list(settings.TRANSLATIONS_ALLOW_LANGS) + ['be'])
        # TODO: https://st.yandex-team.ru/RASPTICKETS-17408
    )
    # национальная версия
    national_version = fields.String(
        load_from='national',
        missing='ru',
        validate=fields.validate.OneOf(settings.ALLOW_NATIONAL_VERSIONS),
    )

    service = fields.String(required=True, validate=validate_service)

    # тестовый контекст
    test_context = fields.String(load_from='testcontext')

    @post_load
    def compose_passengers(self, item):
        item['passengers'] = {
            'adults': item.pop('adults'),
            'children': item.pop('children'),
            'infants': item.pop('infants'),
        }
        return item

    @validates_schema
    def validate_date_forward(self, data, **kwargs):
        if 'date_forward' not in data:
            raise ValidationError('Empty date_forward', field_names=[ErrorCodes.INCORRECT_DATE_FORWARD])
        date = data['date_forward']
        today = datetime.now().date()
        if date < today:
            raise ValidationError(
                'Bad date_forward (%s): date_forward in the past' % date,
                field_names=[ErrorCodes.INCORRECT_DATE_FORWARD]
            )

        if date > today + timedelta(days=366):
            raise ValidationError(
                'Bad date_forward (%s): date_forward more than a year in the future' % date,
                field_names=[ErrorCodes.INCORRECT_DATE_FORWARD]
            )

    @validates_schema
    def validate_date_backward(self, data, **kwargs):
        date_backward = data.get('date_backward')
        if not date_backward:
            return

        if 'date_forward' in data and date_backward < data['date_forward']:
            raise ValidationError(
                'Bad date_backward (%s): date_backward is earlier than date_forward' % date_backward,
                field_names=[ErrorCodes.DATES_CONFLICT]
            )

        if date_backward > datetime.now().date() + timedelta(days=366):
            raise ValidationError(
                'Bad date_backward (%s): date_backward more than a year in the future' % date_backward,
                field_names=[ErrorCodes.INCORRECT_DATE_BACKWARD]
            )


class InitQuerySchema(QueryParamsSchema):
    partner_codes = fields.String(required=False)
    wizard_caches = fields.String(required=False)

    custom_store_time = fields.Integer(
        required=False,
        validate=fields.validate.Range(min=5 * 60, max=1440 * 60)
    )


class JsonEncodedNested(fields.Nested):
    def _serialize(self, nested_obj, attr, obj):
        serialized = super(JsonEncodedNested, self)._serialize(nested_obj, attr, obj)
        return json.dumps(serialized)

    def _deserialize(self, value, attr, data):
        try:
            if value:
                value = json.loads(value)
        except (ValueError, TypeError) as e:
            raise ValidationError(repr(e), data=data)
        return super(JsonEncodedNested, self)._deserialize(value, attr, data)


# Redirect


class _OrderDataSchema(Schema):
    partner = fields.Str(required=True)


class _YaUserSchema(Schema):
    uid = fields.Str()
    login = fields.Str()


class _DjangoUserSchema(Schema):
    username = fields.Str()
    is_staff = fields.Bool()


class _UserInfoSchema(Schema):
    yandexuid = fields.Str()
    passportuid = fields.Str()
    userip = fields.Str()

    yauser = fields.Nested(_YaUserSchema)
    django_user = fields.Nested(_DjangoUserSchema)


class RedirectDataInputSchema(QidInputSchema):
    lang = fields.String(required=True)
    order_data = JsonEncodedNested(_OrderDataSchema)
    user_info = JsonEncodedNested(_UserInfoSchema)
