# -*- coding: utf-8 -*-
from django.conf import settings

import logging

from travel.avia.library.python.common.models.geo import Country
from travel.avia.library.python.common.models.schedule import Company
from travel.avia.library.python.shared_flights_client.client import SharedFlightsClient
from travel.avia.library.python.ticket_daemon.caches.operating_flights.cache import OperatingFlightProvider
from travel.avia.library.python.ticket_daemon.memo import memoize
from travel.avia.ticket_daemon_api.jsonrpc.lib import feature_flags
from travel.avia.ticket_daemon_api.jsonrpc.lib.caches import default_cache
from travel.avia.ticket_daemon_api.jsonrpc.lib.operating_checker import (
    OperatingFlightChecker,
    OperatingFlightCheckException
)
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo import get_station_by_iata_or_sirena, get_station_by_id

logger = logging.getLogger(__name__)

pobeda_flight = OperatingFlightChecker(
    operating_flight_provider=OperatingFlightProvider(SharedFlightsClient(), default_cache),
    airline_id=Company.POBEDA_ID,
    heuristics=lambda number: number.startswith('SU 5') and len(number) == 7
)


@memoize()
def _get_boy_disabled_station_ids():
    if not settings.BOY_DISABLED_DEPARTURE_AIRPORT_CODES:
        return set()
    stations = (
        get_station_by_iata_or_sirena(code) for code in settings.BOY_DISABLED_DEPARTURE_AIRPORT_CODES if code
    )
    return set([station.id for station in stations if station])


def _get_flight_ids_from_variant(variant):
    variant_route_forward = variant['route'][0]
    variant_route_backward = variant['route'][1]
    return variant_route_forward + variant_route_backward


def _flights_are_domestic(flights_iterable):
    stations = (
        get_station_by_id(station_id) for flight in flights_iterable for station_id in (flight['from'], flight['to'])
    )
    return all(station and station.country_id == Country.RUSSIA_ID for station in stations)


def _flights_are_from_boy_disabled_stations(flights_iterable):
    boy_disabled_station_ids = _get_boy_disabled_station_ids()
    return any(flight['from'] in boy_disabled_station_ids for flight in flights_iterable)


def _variant_is_direct(variant):
    variant_route_forward = variant['route'][0]
    variant_route_backward = variant['route'][1]
    return len(variant_route_forward) == 1 and len(variant_route_backward) <= 1


def is_boy_enabled_for_variant(variant, flights, request_params, partner, query, ab_flags):
    """
    :type variant: dict
    :type flights: dict
    :type request_params: dict
    :type partner: basestring
    :type query: travel.avia.ticket_daemon_api.jsonrpc.query.Query
    :type ab_flags: dict
    """
    if partner != settings.AFL_PARTNER_CODE:
        return False  # В самом начале во избежании пустой работы дальше

    if (
        request_params.get('service') == settings.TRAVEL_APP_SERVICE_NAME
        and not feature_flags.afl_feature_flags.boy_travel_app_afl_enable(ab_flags)
    ):
        return False

    if feature_flags.afl_feature_flags.boy_afl_not_business(ab_flags) and query.klass == 'business':
        return False

    utm_source_is_empty = request_params.get('utm_source', '') == ''
    service_direct_redirect = (
        request_params.get('wizard_redir_key') is None
        and request_params.get('unisearch_redir_key') is None
        and request_params.get('utm_medium', '') != 'redirect'
        and request_params.get('utm_content', '') != 'redirect'
    )

    _username = request_params.get('user_info', {}).get('yauser', {}).get('login')

    is_closed_beta_user = _username and (
        _username in settings.BOOKING_TESTERS
        or (feature_flags.boy_granted_to_aeroflot() and _username in settings.BOOKING_TESTERS_AEROFLOT)
    )

    flights_iter = (flights[flight_id] for flight_id in _get_flight_ids_from_variant(variant))
    if _flights_are_from_boy_disabled_stations(flights_iter):
        return False

    def pobeda_operates_iter():
        for flight_id in _get_flight_ids_from_variant(variant):
            try:
                yield pobeda_flight.is_operating(flights[flight_id])
            except OperatingFlightCheckException:
                yield False  # assume pobeda is not operating if anything is wrong

    return is_redirect_as_book_on_yandex(
        request_params,
        partner,
        query,
        utm_source_is_empty,
        service_direct_redirect,
        variant,
        flights_iter,
        is_closed_beta_user,
        pobeda_operates_iter(),
        ab_flags,
    )


def is_redirect_as_book_on_yandex(
    params,
    partner,
    query,
    utm_source_is_empty,
    service_direct_redirect,
    variant,
    flights_iter,
    is_closed_beta_user,
    pobeda_operates_iter,
    ab_flags,
):
    return params.get('book_on_yandex', True) and (
        partner == settings.AFL_PARTNER_CODE
        and feature_flags.afl_feature_flags.boy_afl_enable(ab_flags)
        and (
            (feature_flags.afl_feature_flags.boy_touch_afl_enable(ab_flags) and query.is_mobile)
            or (feature_flags.afl_feature_flags.boy_desktop_afl_enable(ab_flags) and not query.is_mobile)
        )
        and (not feature_flags.afl_feature_flags.boy_afl_only_service(ab_flags) or utm_source_is_empty)
        and (not feature_flags.afl_feature_flags.boy_afl_service_all(ab_flags) or service_direct_redirect)
        and (not feature_flags.afl_feature_flags.boy_afl_only_direct(ab_flags) or _variant_is_direct(variant))
        and (not feature_flags.afl_feature_flags.boy_afl_only_domestic(ab_flags) or _flights_are_domestic(flights_iter))
        and (not feature_flags.afl_feature_flags.boy_afl_adults_only(ab_flags) or query.only_adults)
        and (not feature_flags.afl_feature_flags.boy_afl_only_beta_users(ab_flags) or is_closed_beta_user)
        and (feature_flags.afl_feature_flags.boy_afl_codeshares_with_pobeda(ab_flags) or not any(pobeda_operates_iter))
    )
