# -*- coding: utf-8 -*-
import logging
import re

from marshmallow import Schema, fields, post_load, validates_schema
from marshmallow.exceptions import ValidationError

from travel.avia.library.python.common.models.base_partner_model import PARTNER_CODE_REGEX

from travel.avia.ticket_daemon_api.environment import YANDEX_ENVIRONMENT_TYPE
from travel.avia.ticket_daemon_api.jsonrpc.handlers.scheme import QueryParamsSchema, JsonEncodedNested, QidInputSchema, validate_service
from travel.avia.ticket_daemon_api.jsonrpc.lib.internal_daemon_client import InternalDaemonClient
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.order import get_partner_by_code
from travel.avia.ticket_daemon_api.jsonrpc.views import UserInfoSchema


log = logging.getLogger(__name__)


def getTestContext(fields):
    test_context = None
    if YANDEX_ENVIRONMENT_TYPE != 'production':
        test_context = fields.String(load_from='testcontext', required=False)
    return test_context


class SearchResultsInputSchema(QidInputSchema):
    cont = fields.Integer(required=True)
    allow_portional = fields.Bool(required=False, missing=True)
    ignore_outdated = fields.Bool(load_from='ignoreOutdated', required=False, missing=False)
    user_info = JsonEncodedNested(UserInfoSchema, required=False)
    # тестовый контекст
    test_context = getTestContext(fields)
    service = fields.String(required=True, validate=validate_service)


class FlightWithDateTime(Schema):
    number = fields.String(validate=fields.validate.Regexp(ur'\w{2} \d{1,5}', re.U))
    departure_datetime = fields.DateTime(format='%Y-%m-%dT%H:%M')
    fare_code = fields.String(required=False)


class FlightSeriesField(fields.String):
    default_error_messages = {'invalid': 'Not a valid flight list'}

    def _serialize(self, value, attr, obj):
        raise NotImplementedError

    def _deserialize(self, value, attr, data):
        value = super(FlightSeriesField, self)._deserialize(value, attr, data)
        if not value:
            return []
        try:
            return self._flight_with_datetime(value)
        except ValueError as e:
            log.warning('%r %r', e, value)
            self.fail('invalid')

    def _flight_with_datetime(self, value):
        row_data = []
        for f in value.split(','):
            number, departure_datetime = f.split('.')
            row_data.append(
                {
                    'number': number.replace('+', ' '),
                }
            )
            if departure_datetime:
                row_data[-1]['departure_datetime'] = departure_datetime

        return FlightWithDateTime(strict=True, many=True).load(row_data).data


class PartnerCode(fields.String):
    def __init__(self, validate_exists=False, **kwargs):
        super(PartnerCode, self).__init__(self, **kwargs)

        self._validate_exists = validate_exists
        self.code_validator = fields.validate.Regexp(PARTNER_CODE_REGEX, error='Wrong partner code format')

    def _deserialize(self, value, attr, data):
        code = super(PartnerCode, self)._deserialize(value, attr, data)

        self.code_validator(code)
        if self._validate_exists:
            _validate_partner_code(code)

        return code


class PartnersListField(fields.List):
    def __init__(self, **kwargs):
        super(PartnersListField, self).__init__(PartnerCode(), **kwargs)

    def _serialize(self, value, attr, obj):
        raise NotImplementedError

    def _deserialize(self, value, attr, data):
        if value:
            return [self.container.deserialize(each) for each in value.split(',')]


class RoutesScheme(Schema):
    # маршрут туда: [ (flight_number, flight_departure_datetime), ... ] // [flight_key]
    forward = FlightSeriesField(required=True)

    # маршрут обратно: [ (flight_number, flight_departure_datetime), ... ] // [flight_key]
    backward = FlightSeriesField(missing=None)


class OrderParamsSchema(QueryParamsSchema, RoutesScheme):
    partners = PartnersListField(missing=[])
    partner_codes = fields.String(load_from='partners', required=False)
    ignore_outdated = fields.Bool(load_from='ignoreOutdated', required=False, missing=False)
    max_age = fields.Integer(load_from='maxAge', required=False, missing=None)
    user_info = JsonEncodedNested(UserInfoSchema, required=False)
    variant_tag = fields.String(required=False)
    # тестовый контекст
    test_context = getTestContext(fields)

    @validates_schema
    def validates_flights_idxs(self, data):
        if not data.get('fare_group'):
            return
        len_flights = len(data['forward']) + len(data['backward'] or [])
        for fare in data['fare_group']:
            for j in fare['flight_idxs']:
                if j >= len_flights:
                    raise ValidationError('Wrong flight index in fare_group')

    @post_load
    def denormalize_flights_idxs(self, item):
        if not item.get('fare_group'):
            return item

        forward = item['forward']
        backward = item['backward'] or []

        for fare in item['fare_group']:
            fare['flights'] = {
                # 'forward': [flights[j] for j in sorted(fare['flight_idxs']) if j < len(forward)],
                'forward': [flight for j, flight in enumerate(forward) if j in fare['flight_idxs']],
                # 'backward': [flights[j] for j in sorted(fare['flight_idxs']) if j >= len(forward)],
                'backward': [
                    flight for j, flight in enumerate(backward, start=len(forward)) if j in fare['flight_idxs']
                ],
            }
        return item


def _validate_partner_code(partner):
    if get_partner_by_code(partner) is None:
        raise ValidationError('Unknown partner code "%s"' % partner)


class RedirectDataInputSchema(QueryParamsSchema, RoutesScheme, QidInputSchema):
    partner = PartnerCode(required=True, validate_exists=True)
    revise_price_value = fields.Float(required=False)
    revise_price_currency = fields.String(required=False)
    utm_source = fields.String(required=False)
    utm_campaign = fields.String(required=False)
    utm_medium = fields.String(required=False)
    utm_content = fields.String(required=False)
    wizard_redir_key = fields.String(required=False)
    wizard_flags = fields.String(required=False)
    user_info = JsonEncodedNested(UserInfoSchema, required=True)
    with_baggage = fields.Boolean(required=False, missing=None)
    book_on_yandex = fields.Boolean(required=False, default=True, missing=True)
    tariff_sign = fields.String(required=False)
    marker = fields.String(required=False)
    variant_test_context = fields.String(
        required=False,
        missing=None,
        load_from=InternalDaemonClient.BOY_VARIANT_TEST_CONTEXT_PARAM,
    )
    fare_families_hash = fields.String(required=False)
    avia_brand = PartnerCode(required=False, validate_exists=True)


class SearchFlightsInputSchema(QidInputSchema):
    partners = PartnersListField(required=True)
