# -*- coding: utf-8 -*-
from typing import Dict, Any

from django.conf import settings
from flask import request

import logging
from collections import defaultdict
from itertools import chain
from operator import itemgetter

from travel.avia.library.python.avia_data.models import AviaCompany

from travel.avia.ticket_daemon_api.jsonrpc.carry_on_size_bucket import get_carry_on_bucket_size
from travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.boy_helpers import is_boy_enabled_for_variant
from travel.avia.ticket_daemon_api.jsonrpc.lib import feature_flags
from travel.avia.ticket_daemon_api.jsonrpc.lib.baggage import info_from_key
from travel.avia.ticket_daemon_api.jsonrpc.lib.currencies import CurrencyConverter
from travel.avia.ticket_daemon_api.jsonrpc.lib.flights import IATAFlight
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.variants_fabric import AbstractApiVariants
from travel.avia.ticket_daemon_api.jsonrpc.lib.tariff_serializer import TariffSerializer
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.avia_company import (
    flight_ratings_by_numbers,
    get_alliances_by_ids,
    get_aviacompanies_by_company_ids,
    get_companies_by_ids,
    get_company_tariffs_by_ids,
    get_count_review_by_number,
    get_avia_company_by_id,
    get_company_by_id,
)
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo import (
    get_airport_ids_by_settlement_id,
    get_settlements_by_ids,
    get_station_by_id,
    get_stations_by_ids,
    get_stations_type_by_id,
    station_iata,
    transport_type_codes,
)
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.order import get_partner_by_code

log = logging.getLogger(__name__)


def get_flight_by_key(results):
    # type: (Dict[basestring, AbstractApiVariants])->Dict[basestring, dict]

    def merge_operating_flight(flight1, flight2):
        # type: (Dict[basestring, Any])->Any
        operating1 = flight1.get('operating')
        operating2 = flight2.get('operating')
        return operating1 or operating2

    flight_by_key = {}
    for res in results.itervalues():
        for key, flight in res.flights.iteritems():
            if key in flight_by_key:
                flight_by_key[key]['operating'] = merge_operating_flight(flight_by_key[key], flight)
                continue
            flight_by_key[key] = flight
    return flight_by_key


def complete_results(results, request_params, query, progress=None):
    """
    :type results: typing.Dict[basestring, travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.variants_fabric.AbstractApiVariants]
    :type request_params: typing.Dict[]
    :type query: travel.avia.ticket_daemon_api.jsonrpc.query.Query
    :type progress: travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.views.Progress
    :return:
    """
    flight_by_key = get_flight_by_key(results)
    flights = flight_by_key.values()
    # _fill_t_model(flights, query.lang) # todo RASPTICKETS-9275
    reference = get_reference(results, flights)

    variants = get_result_variants(results, query.national_version)
    _fill_hybrid(variants, flights)
    ab_flags = request.environ['ab_experiment_flags']
    _fill_book_on_yandex_availability(variants, flight_by_key, request_params, query, ab_flags)
    return {
        'reference': reference,
        'variants': variants,
        'progress': progress and dict(progress._asdict()),
    }


def _flights_companies(flights):
    companies = set()
    for flight in flights:
        companies.add(flight['company'])
        if flight.get('operating'):
            companies.add(flight['operating']['company'])
    return companies


def _flights_avia_companies(flights):
    avia_companies = set()
    for flight in flights:
        avia_companies.add(flight['aviaCompany'])
        if flight.get('operating'):
            avia_companies.add(flight['operating']['company'])
    return avia_companies


def _flights_numbers(flights):
    flight_numbers = set()
    for flight in flights:
        flight_numbers.add(flight['number'])
        if flight.get('operating'):
            flight_numbers.add(flight['operating']['number'])
    return flight_numbers


def get_reference(results, flights):
    airport_ids_by_settlement_id = get_airport_ids_by_settlement_id()
    count_review_by_number = get_count_review_by_number()

    stations = get_stations_by_ids(set(map(itemgetter('from'), flights)) | set(map(itemgetter('to'), flights)))
    settlements = get_settlements_by_ids({s.settlement_id for s in stations if s.settlement_id})
    companies = get_companies_by_ids(_flights_companies(flights))
    flights_numbers = _flights_numbers(flights)

    return {
        'flights': flights,
        'companies': companies,
        'aviaCompanies': get_aviacompanies_by_company_ids(_flights_avia_companies(flights)),
        'companyTariffs': get_company_tariffs_by_ids(set(map(itemgetter('companyTariff'), flights))),
        'stations': stations,
        'alliances': get_alliances_by_ids({c.alliance_id for c in companies if c.alliance_id}),
        'ratings': flight_ratings_by_numbers(flights_numbers),
        'reviewsCount': {
            number: count_review_by_number[number] for number in flights_numbers if number in count_review_by_number
        },
        'baggageTariffs': baggage_tariffs(results),
        'fareFamilies': fare_families(results),
        'settlements': settlements,
        'partners': map(get_partner_by_code, results),
        'settlementToCountStations': {
            s.id: len(airport_ids_by_settlement_id[s.id]) for s in settlements if s.id in airport_ids_by_settlement_id
        },
    }


def get_result_variants(results, national_version='ru'):
    ural_airlines_plus_2022 = feature_flags.ural_airlines_plus_2022()
    prices_by_route = defaultdict(list)
    rates = CurrencyConverter(national_version)

    for p_code, vs in results.iteritems():
        partner = get_partner_by_code(p_code)
        if partner is None:
            continue
        charter = p_code in settings.CHARTER_PARTNERS
        is_aviacompany = partner.is_aviacompany
        for variant in vs.variants:
            tariff = variant['tariff']
            v = {
                'baggage': variant['baggage'],
                'tariff': tariff,
                'tariff_sign': TariffSerializer.serialize(variant['tariff'], variant.get('created')),
                'partnerCode': p_code,
                'queryTime': vs.query_time,
            }

            if tariff:
                currency_from = tariff.get('currency')
                currency_to = settings.AVIA_NATIONAL_CURRENCIES.get(national_version)
                value = tariff.get('value')
                if currency_from and value:
                    value_in_national_currency = rates.convert_value(value, currency_from, currency_to)
                    if value_in_national_currency:
                        v['tariffNational'] = {
                            'currency': currency_to,
                            'value': value_in_national_currency,
                        }

            if feature_flags.fill_fare_family_enabled():
                v['fare_families'] = [
                    [ff and ff['key'] for ff in direction_ff] for direction_ff in variant.get('fare_families', [])
                ]
                v['fare_families_hash'] = variant.get('fare_families_hash')
                v['fare_codes'] = (variant.get('fare_codes'),)  # пока оставил для отладки, так он пока не нужен фронту.

            if variant['charter'] is not None or charter:
                # todo я бы эту логику перенес на уровень сохранения результатов
                v['charter'] = charter or variant['charter']

            if is_aviacompany:
                v['fromCompany'] = is_aviacompany

            if 'selfconnect' in variant:
                v['selfconnect'] = variant['selfconnect']

            if 'promo' in variant:
                v['promo'] = variant['promo']

            if 'price_category' in variant:
                v['priceCategory'] = variant['price_category']

            if ural_airlines_plus_2022:
                try:
                    v['plusPoints'] = variant['yandex_plus_promo']['total']
                except (KeyError, TypeError):
                    pass

            prices_by_route[tuple(map(tuple, variant['route']))].append(v)

    return {
        'fares': _fares_json(prices_by_route),
    }


def _fares_json(prices_by_route):
    return [
        {
            'route': route,
            'prices': prices,
        }
        for route, prices in prices_by_route.iteritems()
    ]


def baggage_tariffs(results):
    baggage_keys = set()
    for vs in results.itervalues():
        for v in vs.variants:
            baggage_keys.update(chain.from_iterable(v['baggage']))
    return {key: info_from_key(key) for key in baggage_keys}


def fare_families(results):
    result_fare_families = {}
    if not feature_flags.fill_fare_family_enabled():
        return result_fare_families
    for vs in results.itervalues():
        for v in vs.variants:
            for ff in chain.from_iterable(v.get('fare_families', [])):
                if ff and ff['key'] not in result_fare_families:
                    result_fare_families[ff['key']] = ff
            # fare_families_keys.update(chain.from_iterable(v['fare_families']))
    # return {key: info_from_key(key) for key in fare_families_keys} # todo: можно по ключу формировать описание тарифа, но пока проще так
    return result_fare_families


def _fill_book_on_yandex_availability(variants, flights, request_params, query, ab_flags):
    for v in variants['fares']:
        for price in v['prices']:
            price['boy'] = is_boy_enabled_for_variant(v, flights, request_params, price['partnerCode'], query, ab_flags)


def _fill_hybrid(variants, flights):
    cost_type_by_flight_key = {}

    for flight in flights:
        ac = get_avia_company_by_id(flight['aviaCompany'])
        if ac:
            cost_type_by_flight_key[flight['key']] = ac.cost_type

    for v in variants['fares']:
        company_types = {cost_type_by_flight_key.get(flight_key) for flight_key in chain.from_iterable(v['route'])}
        v['hybrid'] = AviaCompany.HYBRID_COST in company_types and AviaCompany.LOW_COST not in company_types


def serialize_results_flights(results):
    """
    :type results: Dict[basestring, jsonrpc.lib.result.collector.variants_fabric.AbstractSaasVariants]
    :rtype: list
    """
    completed_flights = []
    for p_code, api_variants in results.iteritems():
        partner = get_partner_by_code(p_code)
        from_company = partner and partner.is_aviacompany or False
        for f in api_variants.flights.values():
            company = get_company_by_id(f['company'])
            station_from = get_station_by_id(f['from'])
            station_to = get_station_by_id(f['to'])

            completed_flights.append(
                {
                    'airline_rasp_id': company.id if company else None,
                    'airline_code': company.iata or company.sirena_id if company else None,
                    'flight_number': f['number'],
                    'station_from_rasp_id': station_from.id,
                    'station_from_code': station_iata(station_from) or station_from.sirena_id,
                    'station_from_tz_name': station_from.time_zone,
                    'station_to_rasp_id': station_to.id,
                    'station_to_code': station_iata(station_to) or station_to.sirena_id,
                    'station_to_tz_name': station_to.time_zone,
                    'departure_local': f['departure']['local'] if f['departure'] else None,
                    'arrival_local': f['arrival']['local'] if f['arrival'] else None,
                    'partner_code': p_code,
                    'from_company': from_company,
                }
            )

    return completed_flights


def serialize_results(datum, q, cont=None):
    reference = datum['reference']

    def serialize_datetime(dt_dict):
        if not dt_dict:
            return None
        return {
            'local': (dt_dict['local'] if isinstance(dt_dict['local'], basestring) else dt_dict['local'].isoformat()),
            'offset': dt_dict['offset'],
            'tzname': dt_dict['tzname'],
        }

    result_reference = {
        # 'itineraries': reference['itineraries'],  There will be no itineraries
        'flights': [
            {
                'key': f['key'],
                'number': f['number'],
                'company': f['company'],
                'from': f['from'],
                'to': f['to'],
                'aviaCompany': f['aviaCompany'],
                'companyTariff': f['companyTariff'],
                'departure': serialize_datetime(f['departure']),
                'arrival': serialize_datetime(f['arrival']),
                'tModel': f.get('tModel'),
                'url': IATAFlight.url(f['number'], f['departure'], q.national_version),
                'operating': f.get('operating'),
            }
            for f in reference['flights']
        ],
        'alliances': [
            {
                'id': a.id,
                'title': a.title,
            }
            for a in reference['alliances']
        ],
        'aviaCompanies': [
            {
                'id': c.id,
                'costType': c.cost_type,
                'carryonWidth': c.carryon_width,
                'carryonHeight': c.carryon_height,
                'carryonLength': c.carryon_length,
                'baggageRules': c.baggage_rules,
                'baggageRulesUrl': c.baggage_rules_url,
                'baggageDimensionsSum': c.baggage_dimensions_sum,
                'carryonSizeBucket': get_carryon_size_bucket(c.carryon_width, c.carryon_height, c.carryon_length),
            }
            for c in reference['aviaCompanies']
        ],
        'companies': [
            {
                'id': c.id,
                'url': c.url,
                'color': c.logo_bgcolor,
                'alliance': c.alliance_id,
                'title': c.get_title(lang=q.lang),
                'logoSvg': c.get_logo(),
                'logoPng': c.get_png_logo(),
            }
            for c in reference['companies']
        ],
        'companyTariffs': [
            {
                'id': t.id,
                'baggageAllowed': t.baggage_allowed,
                'carryon': t.carryon,
                'baggageNorm': t.baggage_norm,
                'carryonNorm': t.carryon_norm,
                'published': t.published,
            }
            for t in reference['companyTariffs']
        ],
        'partners': [
            {
                'id': p.id,
                'code': p.code,
                'title': p.get_title(lang=q.lang, national_version=q.national_version),
                'logoSvg': p.get_logo(national_version=q.national_version),
                'logoPng': p.get_png_logo(national_version=q.national_version),
                'siteUrl': p.site_url or None,
            }
            for p in reference['partners']
            if p is not None
        ],
        'ratings': [
            {
                'number': r.number,
                'delayedLess30': r.delayed_less_30,
                'delayed3060': r.delayed_30_60,
                'delayed6090': r.delayed_60_90,
                'delayedMore90': r.delayed_more_90,
                'canceled': r.canceled,
                'scores': r.scores,
            }
            for r in reference['ratings']
        ],
        'reviewsCount': reference['reviewsCount'],
        'settlementToCountStations': reference['settlementToCountStations'],
        'stations': [
            {
                'id': s.id,
                'title': s.get_title(lang=q.lang),
                'code': station_iata(s) or s.sirena_id,
                'phraseTo': s.get_to_title(lang=q.lang),
                'phraseFrom': s.get_from_title(lang=q.lang),
                'phraseIn': s.get_in_title(lang=q.lang),
                'preposition': get_preposition(s, q.lang),
                'tType': transport_type_codes().get(s.t_type_id),
                'stationType': {
                    'prefix': get_stations_type_by_id(s.station_type_id).get_prefix(lang=q.lang),
                    'title': get_stations_type_by_id(s.station_type_id).get_title(lang=q.lang),
                },
                'settlement': s.settlement_id,
            }
            for s in reference['stations']
        ],
        'settlements': [
            {
                'id': s.id,
                'countryId': s.country_id,
                'title': s.get_title(lang=q.lang),
                'phraseFrom': s.get_from_title(lang=q.lang),
                'phraseTo': s.get_to_title(lang=q.lang),
                'phraseIn': s.get_in_title(lang=q.lang),
                'preposition': get_preposition(s, q.lang),
                'code': s.iata or s.sirena_id,
            }
            for s in reference['settlements']
        ],
        'baggageTariffs': reference['baggageTariffs'],
        'fareFamilies': reference['fareFamilies'],
    }

    result = {
        'reference': result_reference,
        'variants': datum['variants'],
        'cont': cont,
        'progress': datum['progress'],
        'marker': datum.get('marker'),
    }

    if 'partners' in datum:
        result['partners'] = datum['partners']

    if 'partnersQueryingInfo' in datum:
        result['partnersQueryingInfo'] = {k: v.to_dict() for k, v in datum['partnersQueryingInfo'].iteritems()}

    return result


def get_preposition(settlement_or_station, lang):
    if not lang or lang == 'ru':
        return settlement_or_station.title_ru_preposition_v_vo_na or u'в'
    return ''


def get_carryon_size_bucket(carryon_width, carryon_height, carryon_length):
    max_part_size = 0
    for value in [carryon_width, carryon_height, carryon_length]:
        if value > max_part_size:
            max_part_size = value
    return get_carry_on_bucket_size(max_part_size)
