# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from itertools import chain

import ujson
from django.conf import settings
from flask import Blueprint, request
from typing import Dict, Tuple, List
from werkzeug.exceptions import BadRequest, GatewayTimeout, NotImplemented as NotImplementedError

from travel.avia.library.python.ticket_daemon.query_validation.validation import validate_query, ValidationError
from travel.avia.library.python.common.models.geo import Country
from travel.avia.library.python.ticket_daemon.ydb.banned_variants.cache import BannedVariantsCache
from travel.avia.library.python.ticket_daemon.ydb.banned_variants.data import BannedVariantParams
from travel.avia.library.python.ticket_daemon.ydb.django import utils as django_ydb_utils

from travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.boy_helpers import is_boy_enabled_for_variant
from travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.serialization import (
    complete_results,
    serialize_results_flights,
    serialize_results,
)
from travel.avia.ticket_daemon_api.jsonrpc.handlers.v3.scheme import (
    OrderParamsSchema,
    RedirectDataInputSchema,
    SearchResultsInputSchema,
    QueryParamsSchema,
    SearchFlightsInputSchema,
    QidInputSchema,
)
from travel.avia.ticket_daemon_api.jsonrpc.lib import feature_flags
from travel.avia.ticket_daemon_api.jsonrpc.lib import result as resultlib
from travel.avia.ticket_daemon_api.jsonrpc.lib.baggage import with_baggage
from travel.avia.ticket_daemon_api.jsonrpc.lib.flights import IATAFlight, Variant
from travel.avia.ticket_daemon_api.jsonrpc.lib.request_timer import RequestTimer
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector import variants_fetcher as fetcherlib
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.test_context import parse_test_context, parse_test_context_proto
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.variants_fabric import AbstractApiVariants
from travel.avia.ticket_daemon_api.jsonrpc.lib.tariff_serializer import TariffSerializer
from travel.avia.ticket_daemon_api.jsonrpc.lib.internal_daemon_client import (
    internal_daemon_client,
    InternalDaemonException,
)
from travel.avia.ticket_daemon_api.jsonrpc.lib.partner_timeout_provider import (
    partner_redirect_timeout_provider,
    UnknownPartnerError,
    DisabledPartnerError,
)
from travel.avia.ticket_daemon_api.jsonrpc.lib.redirect import url_add_openstat_marker
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.continuation import get_cont_details, create_next_cont
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import skip_None_values
from travel.avia.ticket_daemon_api.jsonrpc.lib.yt_loggers.api_variants import api_variants_logger
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.geo import get_settlements_by_ids
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.order import get_partner_by_code
from travel.avia.ticket_daemon_api.jsonrpc.models_utils.price_list import price_list
from travel.avia.ticket_daemon_api.jsonrpc.query import Query, init_search
from travel.avia.ticket_daemon_api.jsonrpc.response_models.partner_querying_info import PartnerQueryingInfo
from travel.avia.ticket_daemon_api.jsonrpc.views import clean_query_from_qid, jsend_view

log = logging.getLogger(__name__)

api = Blueprint('api3', __name__)

Progress = namedtuple('Progress', ['current', 'all'])


@api.route('/results/<qid>/<int:cont>')
@api.route('/results/<qid>/<int:cont>/<int:allow_portional>')
@jsend_view
def results(**kwargs):
    params, errors = SearchResultsInputSchema().load(dict(request.values.items(), **kwargs))
    if errors:
        log.warning('Search results. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _result(params)


@api.route('/wizard_results')
@jsend_view
def wizard_results():
    params, errors = QueryParamsSchema().load(request.values)
    if errors:
        log.warning('Wizard results. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _wizard_result(params)


@api.route('/order')
@jsend_view
def order():
    params, errors = OrderParamsSchema().load(request.values)

    if errors:
        log.warning('Order. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _order(params)


@api.route('/redirect')
@jsend_view
def redirect():
    params, errors = RedirectDataInputSchema().load(request.values)
    if errors:
        log.warning('Redirect data. Params are not valid: %r', errors)
        raise BadRequest(repr(errors))
    return _redirect(params)


@api.route('/statuses/<qid>')
@jsend_view
def statuses(**kwargs):
    params, errors = QidInputSchema().load(kwargs)
    if errors:
        log.warning('Search statuses. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _statuses(params)


@api.route('/results_meta')
@jsend_view
def results_meta(**kwargs):
    """
    Response format:
    {
      "status": "success",
      "data": {
        "dohop_359": null,
        "charterok": null,
        "tinkoff": {
          "instant_search_expiration_time": 1556530056,
          "qid": "190422-122733-506.ticket.plane.c54_c213_2019-04-30_None_economy_1_0_0_ru.ru",
          "expire": 1555926756,
          "created": 1555925256
        },
        ...
      }
    }
    """
    params, errors = QueryParamsSchema().load(request.values)
    if errors:
        log.warning('/result_meta. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _results_meta(params)


@api.route('/flights/<qid>/')
@jsend_view
def flights(**kwargs):
    params, errors = SearchFlightsInputSchema().load(dict(request.values.items(), **kwargs))
    if errors:
        log.warning('Flights results. Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})
    return _flights(params)


@api.route('/track/result/', methods=['GET'])
def track_result(**kwargs):
    return internal_daemon_client.track_result(request.args['trid'])


@api.route('/track/start/', methods=['POST'])
def track_start(**kwargs):
    return internal_daemon_client.start_track(request.json)


@api.route('/invalidate_variant', methods=['POST'])
@jsend_view
def invalidate_variant(**kwargs):
    params, errors = OrderParamsSchema().load(request.values)
    if errors:
        log.warning('Params are not valid: %r', errors)
        raise BadRequest({'errors': errors})

    q = Query(**params)
    try:
        validate_query(q)
    except ValidationError as e:
        raise BadRequest(e.data)
    try:
        if 'variant_tag' in params:
            _save_banned_variant(
                query=q,
                partners=params['partners'],
                variant_info=BannedVariantParams(tag=params['variant_tag']),
            )
    except Exception:
        log.exception('Could not save banned variant: %s', params)

    return init_search(
        q,
        ignore_cache=True,
        user_info={},
        test_id=None,
    )


def _result(params):
    q = clean_query_from_qid(params['qid'])

    cont_details = get_cont_details(q.id, params['cont'])

    if cont_details is None:
        raise BadRequest(
            'No continuation details found.\n'
            'Unknown "qid" or some "cont" is being requested more than 60 seconds.\n'
            'Retry search session'
        )

    skip_partner_codes = set(cont_details.skip_partner_codes or []) | settings.BANNED_PARTNERS_FOR_RESULTS

    results, statuses, revisions = resultlib.collect_variants(
        q,
        skip_partner_codes=skip_partner_codes,
        result_revisions=cont_details.revisions,
        allow_portional=params['allow_portional'],
        mode=(
            resultlib.CollectingModes.actual if params['ignore_outdated'] else resultlib.CollectingModes.instant_search
        ),
        test_context=params.get('test_context'),
    )

    in_progress_partners = [p_code for p_code, status in statuses.iteritems() if resultlib.Statuses.in_progress(status)]
    completed_partners = {
        p_code for p_code, status in statuses.iteritems() if not resultlib.Statuses.in_progress(status)
    }

    actual_variants_count = sum(
        [len(result) for p_code, result in results.iteritems() if result.status != resultlib.Statuses.OUTDATED]
    )

    next_cont_details = create_next_cont(
        cont_details,
        skip_partner_codes=completed_partners,
        revisions=revisions,
        in_progress=in_progress_partners,
        variants_count=actual_variants_count,
    )
    log.debug('next cont_details: %r', next_cont_details)

    all_partners_count = len(statuses) + len(cont_details.skip_partner_codes)
    datum = complete_results(
        results, params, q, progress=Progress(all_partners_count - len(in_progress_partners), all_partners_count)
    )
    datum['partners'] = _prepare_statuses_for_frontend(statuses)

    api_variants_logger.log(
        query=q,
        api_results=results,
        continuation=cont_details,
        completed_partners=list(completed_partners),
        statuses=datum['partners'],
    )

    response = serialize_results(datum, q, cont=next_cont_details.cont)
    status_code = 200

    if feature_flags.new_results_statuses():
        if in_progress_partners:
            status_code = 206
        elif next_cont_details.variants_count == 0:
            status_code = 204

    return response, status_code


def _wizard_result(params):
    q = Query(**params)

    _results, qid = resultlib.collect_wizard_variants(q)

    datum = complete_results(_results, params, q)
    # datum['partners'] = _prepare_statuses_for_frontend(statuses)  # todo нужны ли тут статусы?
    datum = serialize_results(datum, q)
    datum['qid'] = qid
    return datum


def _unprocessed_variant_match_route(params):
    route = [
        [IATAFlight.make_flight_tag(f.get('departure_datetime'), f['number']) for f in params[dn] or []]
        for dn in ['forward', 'backward']
    ]

    def unprocessed_variant_match_route(v):
        """
        Так проверка сделана потому что фронтенд не передаёт время прилета,
        но он добавился к ключу что бы избежать багов с "маршрутками"
        """
        vroute = v['route']
        result = (
            len(vroute[0]) == len(route[0])
            and len(vroute[1]) == len(route[1])
            and all(i.startswith(j) for i, j in zip(chain(*vroute), chain(*route)))
        )
        return result

    return unprocessed_variant_match_route


def _order(params):
    q = Query(**params)

    mode = resultlib.CollectingModes.instant_search
    if params['ignore_outdated']:
        mode = resultlib.CollectingModes.actual

    results = None
    statuses = dict()
    test_context = params.get('test_context')
    if test_context:
        tk = parse_test_context_proto(test_context)
        if tk.MockAviaVariants:
            results = dict(parse_test_context(test_context))
            for partner_code, unpacked_variants in results:
                statuses[partner_code] = resultlib.Statuses.DONE

    if results is None:
        results, statuses, _ = resultlib.collect_variants(
            q,
            prefilter=_unprocessed_variant_match_route(params),
            partner_codes=params['partners'],
            mode=mode,
            max_age=params['max_age'],
            test_context=params.get('test_context'),
        )

    datum = complete_results(results, params, q)
    datum['partners'] = _prepare_statuses_for_frontend(statuses)
    datum['partnersQueryingInfo'] = _prepare_querying_verbose_info(results)
    return serialize_results(datum, q)


def _process_redirect_data(redirect_data, variants, params, klass):
    # type: (Tuple[str, dict], dict[str, dict], dict, str) -> Tuple[str, dict]
    """Выбирает redirect data для минимального тарифа, соответствующего параметрам запроса.

    :param redirect_data: распакованная redirect data из ydb
    :param variants: распакованные variants из ydb
    :param params: параметры запроса к ticket-daemon-api
    :param klass: класс обслуживания (вычисляется из параметров запроса к ticket-daemon-api выше)
    """
    redirect_data_dict = dict(redirect_data)
    # на всякий случай пока (RASPTICKETS-20782) оставили старое поведение под флагом
    if not feature_flags.store_min_tariff_per_fare_code():
        partner = params['partner']
        partner_redirect_data = redirect_data_dict.get(partner)
        tags = _make_variant_tags(
            params['forward'],
            params['backward'] or [],
            klass,
            partner,
            params['with_baggage'],
        )
        tag = tags[0]
        if not partner_redirect_data:
            log.warning('No redirect data for partner %s in cache. Params: %r', partner, params)
            return tag, None

        for _tag in tags:
            if _tag in partner_redirect_data['variants']:
                return _tag, partner_redirect_data['variants'][_tag]
        else:
            log.warning('No variant in redirect data for partner %s in cache. Params: %r', partner, params)
            return tag, None

    current_tag = None
    current_redirect_data = None
    current_price_value = None
    current_price_currency = None

    '''
    Как работает fare_families_hash. Тикет-демон сохраняет в YDB варианты и их fare codes.
    Тикет-демон-апи, получив варианты из YDB, на основании fare codes вычисляет для каждого варианта его
    fare family (LITE/STANDARD/MAXIMUM и т.п.)
    Если для получения redirect data фронт передал в тикет-демон-апи значение fare_families_hash - тикет-демон-апи вернёт
    самый дешёвый вариант для этого fare family (то, что разные варианты STANDARD могут иметь отличия по нормам багажа и пр.,
    учитывается при построении fare_families_hash, и самый дешёвый вариант выбирается только по тем вариантам fare family,
    у которых hash совпадает с переданным).
    Если для получения redirect data фронт не передал значения fare_families_hash, будет возвращён самый дешёвый или
    самый дешёвый багажный вариант, в зависимости от значения параметра with_baggage.
    '''
    fare_families_hash = params.get('fare_families_hash')
    should_have_baggage = params.get('with_baggage')

    for partner, partner_variants in variants.iteritems():
        if not partner_variants:
            continue
        partner_redirect_data = redirect_data_dict.get(partner)
        if not partner_redirect_data:
            log.warning('No redirect data for partner %s', partner)
            continue
        for fare_variant in partner_variants.variants:
            if not fare_variant:
                continue
            tag = fare_variant.get('tag')
            tariff = fare_variant.get('tariff')
            if not tag or not tariff:
                log.warning('No fare tag or tariff in variant %s', fare_variant)
                continue
            if fare_families_hash:
                if fare_variant.get('fare_families_hash') != fare_families_hash:
                    continue
            elif should_have_baggage:
                baggage = fare_variant.get('baggage')
                if not baggage or not with_baggage(baggage):
                    continue
            partner_redirect_data_variants = partner_redirect_data.get('variants')
            if not partner_redirect_data_variants:
                continue
            if not partner_redirect_data_variants.get(tag):
                continue
            tariff_currency = tariff.get('currency')
            if current_price_currency and current_price_currency != tariff_currency:
                log.warning('Multiple currencies in variants %s', partner_variants.variants)
                continue
            tariff_value = tariff.get('value')
            if current_price_currency and tariff_value > current_price_value:
                continue
            current_price_value = tariff_value
            current_price_currency = tariff_currency
            current_tag = tag
            current_redirect_data = partner_redirect_data_variants.get(tag)
    return current_tag, current_redirect_data


def _make_variant_tags(forward, backward, klass, partner_code, with_baggage):
    forward_tags = [IATAFlight.make_flight_tag(d.get('departure_datetime'), d['number']) for d in forward]
    backward_tags = [IATAFlight.make_flight_tag(d.get('departure_datetime'), d['number']) for d in backward]
    if with_baggage is None:
        return [
            Variant.make_tag(forward_tags, backward_tags, klass, partner_code, False),
            Variant.make_tag(forward_tags, backward_tags, klass, partner_code, True),
        ]
    else:
        return [Variant.make_tag(forward_tags, backward_tags, klass, partner_code, with_baggage)]


def _fill_query_source(datum, query_source):
    # todo RASPTICKETS-10501
    try:
        assert len(datum['variants']['fares']) == 1
        assert len(datum['variants']['fares'][0]['prices']) == 1
        datum['variants']['fares'][0]['prices'][0]['service'] = query_source
    except Exception:
        log.exception("Can't fill query source %r. %r", query_source, datum)


def serialize_redirect_data(q, datum, redirect_data, partner, order_data):
    serialized = serialize_results(datum, q)

    avia_hosts = settings.AVIA_HOST_BY_NATIONAL_VERSION
    travel_hosts = settings.TRAVEL_HOST_BY_NATIONAL_VERSION
    host = travel_hosts.get(
        q.national_version,
        avia_hosts.get(q.national_version, avia_hosts['ru']),
    )
    if redirect_data:
        serialized.update(
            skip_None_values(
                {
                    'url': url_add_openstat_marker(
                        url=redirect_data.redirect_url(q.is_mobile),
                        partner_code=partner,
                        host=host,
                        domain=q.national_version,
                    ),
                    'post': redirect_data.post,
                }
            )
        )
        _fill_query_source(serialized, redirect_data.query_source)
        serialized['marker'] = redirect_data.get_marker(datum.get('marker'))

    if 'click_price' in datum:
        serialized['click_price'] = datum['click_price']

    if order_data and 'shown_tariff' in order_data:
        serialized['shown_tariff'] = order_data['shown_tariff']

    return serialized


def _fill_order_data(fare_redirect_data, q, params, tariff_sign, variant):
    shown_tariff = TariffSerializer.deserialize(tariff_sign) if tariff_sign else None

    order_data = fare_redirect_data['order_data'].copy()
    order_data['national_version'] = q.national_version
    order_data['lang'] = q.lang
    order_data['partner'] = params['partner']
    order_data['qkey'] = q.qkey
    order_data['qid'] = params['qid']

    order_data['tariff'] = variant.get('tariff', {}).copy()
    order_data['tariff'].update({'price_unixtime': variant['created']}) if order_data['tariff'] else None
    order_data['shown_tariff'] = shown_tariff
    order_data['avia_brand'] = params.get('avia_brand')
    return order_data


def _check_redirect_problems(fare_redirect_data, partner, results, status):
    problems_message = ''
    if fare_redirect_data is None:
        problems_message += 'No redirect data. '
    if partner not in results:
        problems_message += 'No results for partner %s. ' % partner
    if problems_message:
        problems_message += 'Query status: %r' % status
        if status is None:
            raise NotImplementedError(problems_message)
        if resultlib.Statuses.in_progress(status):
            return {'message': problems_message}, 206
        if resultlib.Statuses.is_done(status):
            return {'message': problems_message}, 204
    return None


def _redirect(params):
    timer = RequestTimer(settings.DAEMON_API_REDIRECT_TIMEOUT)

    q = Query(**params)

    collect_mode = resultlib.CollectingModes.all
    if feature_flags.do_not_give_not_ready_redirects() and q.service != 'wizard':
        collect_mode = resultlib.CollectingModes.actual

    partner = params['partner']
    variants_data = fetcherlib.get_variants_and_redirect_data(q.queries[0], [partner], True)

    variants, statuses, _ = resultlib.collect_variants(
        query=q,
        prefilter=_unprocessed_variant_match_route(params),
        partner_codes=[partner],
        mode=collect_mode,
        prefetched_results=fetcherlib.get_variants(variants_data),
    )

    redirect_data = fetcherlib.get_redirect_data(variants_data)
    tag, fare_redirect_data = _process_redirect_data(redirect_data, variants, params, q.klass)
    status = statuses.get(partner)

    if feature_flags.new_redirect_statuses():
        problems_response = _check_redirect_problems(fare_redirect_data, partner, variants, status)
        if problems_response is not None:
            return problems_response
    else:
        if fare_redirect_data is None:
            raise BadRequest('Can not fetch the fare redirect data')
        if partner not in variants:
            raise BadRequest('Can not fetch the result for partner {}'.format(partner))

    variants[partner].filter(lambda v: v['tag'] == tag)
    variant = variants[partner].variants[0] if variants[partner].variants else {}
    tariff_sign = params.get('tariff_sign')

    order_data = _fill_order_data(fare_redirect_data, q, params, tariff_sign, variant)

    additional_data = dict(params)

    ab_flags = request.environ['ab_experiment_flags']
    should_ask_book_on_yandex = is_boy_enabled_for_variant(
        variant, variants[partner].flights, params, partner, q, ab_flags
    )

    redirect_data = None
    if should_ask_book_on_yandex:
        try:
            redirect_data = internal_daemon_client.book(
                order_data=order_data,
                variant=variant,
                flights=variants[partner].flights,
                user_info=params['user_info'],
                additional_data=additional_data,
                utm_source=q.service,
                query_source=fare_redirect_data['query_source'],
                timeout=timer.get_available_timeout(settings.BOOK_ON_YANDEX_REQUEST_TIMEOUT),
                variant_test_context=params['variant_test_context'],
            )
        except InternalDaemonException as e:
            log.warning(e, exc_info=True)

            if feature_flags.return_404_status_code_on_boy_404_response() and e.code == 400:
                response = ujson.loads(e.response)
                if response['status_code'] == 404:
                    return response, 404

            if feature_flags.return_timeout_on_boy_timeout() and e.code == 408:
                log.error("BoY timeout", exc_info=False)
                raise GatewayTimeout("BoY timeout", e.response)

            should_ask_book_on_yandex = False  # fallback to cook_redirect/

    if not should_ask_book_on_yandex:
        redirect_data = internal_daemon_client.redirect(
            order_data=order_data,
            user_info=params['user_info'],
            additional_data=additional_data,
            utm_source=q.service,
            query_source=fare_redirect_data['query_source'],
            timeout=timer.get_available_timeout(get_partner_redirect_timeout(order_data['partner'])),
        )

    datum = complete_results(variants, params, q)
    datum['partners'] = _prepare_statuses_for_frontend(statuses)

    datum['click_price'] = _get_click_price(
        q,
        datum,
        utm_source=params.get('utm_source'),
        utm_medium=params.get('utm_medium'),
        utm_content=params.get('utm_content'),
        wizard_redir_key=params.get('wizard_redir_key') or params.get('unisearch_redir_key'),
        partner_code=partner,  # We can enable or not new pricing for different partners
    )

    if _zero_click_price(datum, partner, q.national_version, params.get('avia_brand')):
        datum['click_price']['price'] = 0.0

    datum['marker'] = params.get('marker')

    return serialize_redirect_data(q, datum, redirect_data, partner, order_data)


def get_partner_redirect_timeout(partner_code):
    # type: (str)->float
    try:
        return partner_redirect_timeout_provider.get(partner_code=partner_code)
    except (DisabledPartnerError, UnknownPartnerError) as e:
        # todo перенести эти валидации на уровень internal TD
        # чтобы по 10 раз это не обрабатывать
        raise BadRequest(e)


def _route(datum):
    first_route = datum['variants']['fares'][0]['route'][0][0]
    last_route = datum['variants']['fares'][0]['route'][0][-1]
    station_from_id = next(flight['from'] for flight in datum['reference']['flights'] if flight['key'] == first_route)
    station_to_id = next(flight['to'] for flight in datum['reference']['flights'] if flight['key'] == last_route)

    settlement_from_id = next(
        station.settlement_id for station in datum['reference']['stations'] if station.id == station_from_id
    )
    settlement_to_id = next(
        station.settlement_id for station in datum['reference']['stations'] if station.id == station_to_id
    )

    return settlement_from_id, settlement_to_id


def _get_click_price(
    q, datum, utm_source=None, utm_medium=None, utm_content=None, wizard_redir_key=None, partner_code=None
):
    settlement_from_id, settlement_to_id = _route(datum)

    price_list_result = price_list().get_click_price(
        settlement_from_id,
        settlement_to_id,
        q.national_version,
        q.date_forward,
        q.date_backward,
        q.adults,
        q.children,
        utm_source,
        utm_medium,
        utm_content,
        wizard_redir_key,
        use_new_pricing=feature_flags.new_pricing_flag_by_partner_code(partner_code),
    )

    price_list_result.update(
        {
            'settlement_from_id': settlement_from_id,
            'settlement_to_id': settlement_to_id,
        }
    )

    return price_list_result


def _statuses(params):
    q = clean_query_from_qid(params['qid'])

    return resultlib.collect_statuses(q.qkey, enabled_partner_codes=q.get_enabled_partner_codes())


def _prepare_statuses_for_frontend(statuses):
    """
    TODO: договориться с фронтендом о формате
    :param statuses:
    :return:
    """
    return {
        p_code: status if resultlib.Statuses.in_progress(status) else resultlib.Statuses.DONE
        for p_code, status in statuses.iteritems()
    }


def _prepare_querying_verbose_info(results):
    # type: (Dict[str, AbstractApiVariants]) -> Dict[str, PartnerQueryingInfo]
    return {pcode: PartnerQueryingInfo(vs.status, vs.query_time, vs.qid) for pcode, vs in results.iteritems()}


def _flights(params):
    q = clean_query_from_qid(params['qid'])

    results, _, _ = resultlib.collect_variants(
        q,
        partner_codes=params['partners'],
        allow_portional=False,
        mode=resultlib.CollectingModes.actual,
    )

    completed_flights = serialize_results_flights(results)

    return {
        'qid': q.id,
        'flights': completed_flights,
        'partners': results.keys(),
    }


def _results_meta(params):
    q = Query(**params)
    return resultlib.get_results_meta(q, q.get_enabled_partner_codes())


def _zero_click_price(datum, partner, national_version, avia_brand):
    # type: (dict, str, str, str) -> bool
    if partner == avia_brand and partner in settings.BANNED_PARTNERS_FOR_RESULTS:
        return True

    if get_partner_by_code(partner).pricing_model == 'cpa':
        return True

    if (
        settings.COVID19_ZERO_REDIRECT_PRICE
        and partner in settings.COVID19_ZERO_REDIRECT_PRICE_PARTNERS
        and national_version in settings.COVID19_ZERO_REDIRECT_PRICE_NATIONAL_VERSIONS
        and any(s.country_id != Country.RUSSIA_ID for s in get_settlements_by_ids(_route(datum)))
    ):
        return True

    return False


def _save_banned_variant(query, partners, variant_info):
    # type: (Query, List[basestring], BannedVariantParams)->None
    if not variant_info:
        return
    if not partners:
        return
    banned_variants = BannedVariantsCache(django_ydb_utils.session_manager)
    for partner in partners:
        # почти всегда в partners будет только один партнёр (см invalidateVariant в AviaTicketDaemonApiClient)
        banned_variants.update_banned_variants(query=query, partner_code=partner, variant_info=variant_info)
