# -*- coding: utf-8 -*-
import logging

from travel.avia.library.python.avia_data.libs.currency.rates.caching import CurrencyRatesCache
from travel.avia.library.python.common.utils.currency_converter import ConverterCache
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import dict_merge


log = logging.getLogger(__name__)

# Apply 3% exchange rate fine to get the same result as the travel-front does:
# https://st.yandex-team.ru/TRAVELAPP-593#625e8c433a125a674c88ccc3
RATE_WITH_EXCHANGE_SURCHARGE = 1.03


class CurrencyConverter(object):

    def __init__(self, national_version):
        # type: (str) -> None
        self.rates = _get_currency_rates(national_version)

    def convert_value(self, value, from_currency, to_currency):
        # type: (float, str, str) -> float
        if from_currency == to_currency:
            return value
        if not self.rates:
            return None
        try:
            return float(value) * RATE_WITH_EXCHANGE_SURCHARGE * self.rates[from_currency] / self.rates[to_currency]
        except Exception as exc:
            log.warning(
                "Couldn't convert tariff(%s->%s): %s",
                from_currency, to_currency, exc
            )
            return None


def _get_currency_rates(national_version):
    # type: (str) -> dict

    rates_caches = filter(None, [
        ConverterCache.load(national_version),
        CurrencyRatesCache.load(national_version),
    ])
    if not rates_caches:
        return None
    return dict_merge(*[cache.rates for cache in rates_caches])
