# -*- coding: utf-8 -*-
from __future__ import absolute_import

import time
from abc import abstractproperty, ABCMeta
from datetime import datetime

import pytz

from travel.avia.library.python.common.utils.date import MSK_TZ


FORMAT_DATE = '%Y-%m-%d %H:%M:%S'
EPOCH = datetime(1970, 1, 1)
EPOCH_DATE = EPOCH.date()


def get_utc_now():
    return pytz.UTC.localize(datetime.utcnow())


def get_msk_now():
    return get_utc_now().astimezone(MSK_TZ)


def unixtime():
    # type: () -> int
    return int(time.time())


def naive_to_timestamp(dt, epoch=datetime(1970, 1, 1)):
    td = dt - epoch
    return int(td.total_seconds())


def aware_to_timestamp(dt):
    utc_naive = dt.replace(tzinfo=None) - dt.utcoffset()
    return naive_to_timestamp(utc_naive)


def utc_offset(naive_dt, tz):
    offset = pytz.timezone(tz).localize(naive_dt).utcoffset()
    return offset.total_seconds() / 60.


class AbstractCachedDateTimeDeserializer(object):
    __metaclass__ = ABCMeta
    FORMAT_DATE = abstractproperty()

    def __init__(self):
        self._cache = {}

    def deserialize(self, raw_datetime):
        if raw_datetime not in self._cache:
            self._cache[raw_datetime] = datetime.strptime(
                raw_datetime, self.FORMAT_DATE
            )
        return self._cache[raw_datetime]


class DateTimeDeserializer(AbstractCachedDateTimeDeserializer):
    FORMAT_DATE = '%Y-%m-%dT%H:%M:%S'


class TicketDaemonFlightDateTimeDeserializer(AbstractCachedDateTimeDeserializer):
    FORMAT_DATE = '%Y-%m-%dT%H:%M:%S'

    def deserialize(self, raw_datetime):
        """
        :param dict raw_datetime: {'local': [str], 'tzname': [str], offset: [int]}
        """
        cache_key = (raw_datetime['local'], raw_datetime['tzname'])
        if cache_key not in self._cache:
            departure_unaware_dt = datetime.strptime(raw_datetime['local'], self.FORMAT_DATE)
            local_tz = pytz.timezone(raw_datetime['tzname'])
            self._cache[cache_key] = local_tz.localize(departure_unaware_dt)
        return self._cache[cache_key]
