# -*- coding: utf-8 -*-
from abc import abstractmethod, ABCMeta

import retrying
from django.db.models import Q, FieldDoesNotExist

from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant
from travel.avia.library.python.common.models.partner import DohopVendor, Partner
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.ticket_daemon.memo import memoize, CacheWithKeyTTL


class IEnabledPartnerProvider(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def get_codes(self, national_version, mobile, for_init_search, is_from_rasp):
        """Достать список включенных партеров/вендоров.
            :param bool national_version: национальная версия
            :param bool mobile: отдать только партнеров, которые имеют мобильные версии
            :param bool for_init_search: отдать партеров, учавствующих в инициализации поиска
            :param bool is_from_rasp: отдать партнеров, включенных в расписаниях

            :rtype: typing.List[str]
        """

    @abstractmethod
    def validate_enabled_field_name(self, national_version, mobile, for_init_search, is_from_rasp):
        """Проверить наличие полей для поиска партнеров"""


class EnabledPartnerProvider(IEnabledPartnerProvider):
    """
    Поставщик включенных партнеров и вендоров
    """
    def __init__(self, environment):
        self._environment = environment

    def validate_enabled_field_name(self, national_version, mobile, for_init_search, is_from_rasp):
        is_from_rasp = self._resolve_variable_for_init_search(for_init_search, is_from_rasp)
        field_name = self._get_enabled_field_name(national_version, is_from_rasp, mobile)
        return self._check_exist_attribute_in_partner(field_name)

    def get_codes(self, national_version, mobile, for_init_search, is_from_rasp):

        is_from_rasp = self._resolve_variable_for_init_search(for_init_search, is_from_rasp)

        return self._get_actual_partners(
            national_version, is_from_rasp, mobile, for_init_search
        )

    @staticmethod
    def _check_exist_attribute_in_partner(attribute):
        try:
            return bool(Partner._meta.get_field(attribute))
        except FieldDoesNotExist:
            return False

    @staticmethod
    def _resolve_variable_for_init_search(for_init_search, variable):
        if for_init_search is False and variable:
            variable = False
        return variable

    @memoize(lambda _self, nv, from_rasp, mobile, for_init_search: (nv, from_rasp, mobile, for_init_search), CacheWithKeyTTL(60))
    @retrying.retry(stop_max_attempt_number=3, wrap_exception=True)
    def _get_actual_partners(self, national_version, from_rasp=False, mobile=False, for_init_search=False):
        partners = self._get_partners(national_version, from_rasp, mobile)

        if 'dohop' in partners:
            partners.remove('dohop')
            partners.extend(self._get_dohop_partners(national_version, from_rasp, mobile))

        if 'amadeus' in partners:
            partners.remove('amadeus')
            partners.extend(self._get_amadeus_partners(national_version, from_rasp, mobile))

        return partners

    def _get_partners(self, national_version, from_rasp, mobile):
        flag = self._get_enabled_field_name(national_version, from_rasp, mobile)

        filters = Q(**{
            't_type_id': 2,
            'can_fetch_by_daemon': True,
            'disabled': False,
            flag: True
        }) | Q(**{
            'can_fetch_by_daemon': True,
            'code': 'dohop',
            flag: True
        })

        partners = list(
            Partner.objects
            .filter(filters)
            .values('code', 'start_unavailability_datetime', 'end_unavailability_datetime'))

        now = self._environment.now()

        def tmp_disabled(p):
            start = p['start_unavailability_datetime']
            end = p['end_unavailability_datetime']

            if start and end:
                # django 1.11.28 начал возвращать даты с таймзоной settings.TIME_ZONE
                if start.replace(tzinfo=None) <= now <= end.replace(tzinfo=None):
                    return True
            return False

        return [
            p['code'] for p in partners
            if not tmp_disabled(p)
        ]

    def _get_dohop_partners(self, national_version, from_rasp, mobile):
        flag = self._get_enabled_field_name(national_version, from_rasp, mobile)
        index_code = 'yandexcom' if national_version == 'com' else 'yandex'

        filters = {
            'enabled': True,
            'index_infos__index__code': index_code,
            flag: True
        }

        return [p['code'] for p in DohopVendor.objects.filter(**filters).values('code')]

    def _get_amadeus_partners(self, national_version, from_rasp, mobile):
        flag = self._get_enabled_field_name(national_version, from_rasp, mobile)

        filters = {
            'enabled': True,
            flag: True
        }

        return [p['code'] for p in AmadeusMerchant.objects.filter(**filters).values('code')]

    def _get_enabled_field_name(self, national_version, from_rasp, mobile):
        service = 'rasp' if from_rasp else 'ticket'

        flag = 'enabled_in_mobile_{}_{}' if mobile else 'enabled_in_{}_{}'
        return flag.format(service, national_version)


enabled_partner_provider = EnabledPartnerProvider(environment)
