# -*- coding: utf-8 -*-
import logging

from django.conf import settings
from flask import request

import feature_flag_client
from travel.avia.library.python.flask_helpers import safe_get_request_environ_value

logger = logging.getLogger('feature_flag_client')

_feature_flags_client = feature_flag_client.Client(
    host=settings.FEATURE_FLAG_HOST,
    service_code='ticket-daemon-api',
    logger=logger
)

if settings.YANDEX_ENVIRONMENT_TYPE == 'development':
    class DevelopmentClient(object):
        DEVELOPMENT_FLAGS = []
        if hasattr(settings, 'DEVELOPMENT_FLAGS'):
            DEVELOPMENT_FLAGS = settings.DEVELOPMENT_FLAGS

        def create_context(self):
            return feature_flag_client.Context(set(
                value.strip() for value in self.DEVELOPMENT_FLAGS
            ))
    _feature_flags_client = DevelopmentClient()

feature_flags_storage = feature_flag_client.Storage(_feature_flags_client)


def get_ab_flags():
    return safe_get_request_environ_value('ab_experiment_flags', default={})


def flag_enabled(flag_name):
    ab_flags = feature_flag_client.ABFlagsContent(get_ab_flags())
    return feature_flags_storage.flag_enabled(flag_name, ab_flags)


def boy_granted_to_aeroflot():
    # type: () -> bool
    """Включает возможность переходить на бронирование для аккаунтов Аэрофлота."""
    return flag_enabled('BOY_GRANTED_TO_AEROFLOT')


def _new_pricing_flag_name_by_partner_code(partner_code):
    # type: (str) -> str
    return 'AVIA_PRICING_2019_{}'.format(partner_code.upper())


def new_pricing_flag_by_partner_code(partner_code):
    # type: (str) -> bool
    """RASPTICKETS-15095: separate pricing flags."""
    return flag_enabled(
        _new_pricing_flag_name_by_partner_code(partner_code)
    )


def replace_kk_with_ru_in_kz():
    # type: () -> bool
    """RASPTICKETS-14858: Поиски с tld=kz lang=kk превращать в tld=kz lang=ru"""
    return flag_enabled('REPLACE_KK_WITH_RU_IN_KZ')


def init_multiple_queries():
    # type: () -> bool
    """RASPTICKETS-14088: Поиск до множества пунктов в демоне"""
    return flag_enabled('INIT_MULTIPLE_QUERIES')


def write_in_redis_cache():
    """
        RASPTICKETS-16545
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_WRITE_IN_REDIS')


def read_from_redis_cache():
    """
        RASPTICKETS-16545
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_READ_FROM_REDIS')


def fill_fare_family_enabled():
    """
    RASPTICKETS-16777 Прокидывать информацию об опциях тарифа в ответ TD-API
    :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_FARE_FAMILIES')


def recreate_ydb_session_pool():
    """
        RASPTICKETS-17610 Ошибки сохранения в YDB
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_RECREATE_YDB_SESSION_POOL')


def new_redirect_statuses():
    """
        RASPTICKETS-16935 Отдавать правильные коды для редиректа и ошибок редиректа
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_NEW_REDIRECT_STATUS_CODES')


def new_results_statuses():
    """
        RASPTICKETS-16819 Отдавать новые статусы для v3/results
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_NEW_RESULTS_STATUS_CODES')


def do_not_give_not_ready_redirects():
    """
        RASPTICKETS-17074 Возвращаем на фронт только актуальные редиректы
        :rtype:bool
    """
    flag = 'DO_NOT_GIVE_NOT_READY_REDIRECTS'
    ab_flags = request.environ.get('ab_experiment_flags', {})
    if ab_flags.get(flag) == '0':
        return False
    return ab_flags.get(flag) == '1' or _do_not_give_not_ready_redirects()


def _do_not_give_not_ready_redirects():
    """
        RASPTICKETS-17074 Возвращаем на фронт только актуальные редиректы
        :rtype:bool
    """
    return flag_enabled('TICKET_DAEMON_API_DO_NOT_GIVE_NOT_READY_REDIRECTS')


def replace_search_to_station_with_search_to_city():
    """
        RASPTICKETS-18936 Обобщение при поиске до станций
        :rtype:bool
    """
    return flag_enabled('TICKET_DAEMON_API_REPLACE_SEARCH_TO_STATION_WITH_S')


def return_404_status_code_on_boy_404_response():
    """
        RASPTICKETS-18802 При редиректе на Аэрофлот при ответе 404 от booking api нужно на фронт передавать 404
        :rtype:bool
    """
    return flag_enabled('TICKET_DAEMON_API_THROW_OUT_BOY_404_RESPONSE')


def return_timeout_on_boy_timeout():
    """
        RASPTICKETS-21066 Возвращать gateway timeout, если не достучались до BoY
        :rtype:bool
    """
    return flag_enabled('TICKET_DAEMON_API_TIMEOUT_ON_BOY_TIMEOUT')


def store_min_tariff_per_fare_code():
    """
        RASPTICKETS-20782
        :rtype: bool
    """
    return flag_enabled('TICKET_DAEMON_API_STORE_MIN_TARIFF_PER_FARE_CODE')


def use_fare_families_from_variants_cache():
    """
        RASPTICKETS-22537
        :rtype: bool
    """
    return flag_enabled('TD_API_USE_FARE_FAMILIES_FROM_VARIANTS_CACHE')


def ural_airlines_plus_2022():
    return flag_enabled('URAL_AIRLINES_PLUS_2022')


def use_shared_flights_for_close_dates_direct_flights():
    """
        RASPTICKETS-22453
        :rtype: bool
    """
    return flag_enabled('TD_API_SHOW_CLOSE_DAYS_DIRECT_FLIGHTS_IN_INIT')


class AeroflotBOYFlags(object):
    def boy_afl_enable(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Флаг для включения/выключения потока на собственное бронирование.

        Если выключен, то возвращаем диплинк для кликаута, при любом включенном флаге BOY_AFL_.

        Если включен,
            то возвращаем ссылку на собственное бронирование с учетом состояния других флагов BOY_AFL_.
        """
        return self._is_flag_enabled('BOY_AFL_ENABLE', ab_flags)

    def boy_afl_only_service(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Флаг для ограничения потока на собственное бронирование
            только пользователями сервиса avia.yandex.ru

        Если включен совместно с BOY_AFL_ENABLE,
            то возвращаем ссылку на прямое бронирование в том случае,
            если пользователь пришел с сервиса, utm_source пустой.
        Для остальных источников перехода (utm_source не пустой) отправляем пользователя в Аэрофлот.

        Если выключен,
            то не накладывает дополнительную фильтрацию на возврат ссылок на прямое бронирование.
        """
        return self._is_flag_enabled('BOY_AFL_ONLY_SERVICE', ab_flags)

    def boy_afl_service_all(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-16015: Этот флаг про ограничение бронирования только на
            тех пользователей, которые совершают переход с сервиса, а не
            напрямую из других источников (wizard, sovetnik, rasp)

        Если включен совместно с BOY_AFL_ENABLE,
            то возвращаем ссылку на прямое бронирование в том случае,
            когда wizardRedirKey is NULL and utm_medium != "redirect" and utm_content !="redirect"

        Для остальных источников перехода отправляем пользователя в Аэрофлот.

        Если выключен,
            то не накладывает дополнительную фильтрацию на возврат ссылок на прямое бронирование.
        """
        return self._is_flag_enabled('BOY_AFL_SERVICE_ALL', ab_flags)

    def boy_afl_only_direct(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Флаг ограничивающий поток только рейсами без пересадок.

        Если включен совместно с BOY_AFL_ENABLE,
            то возвращает ссылку на прямое бронирование в том случае,
            если в вариантах перелета и в прямом и в обратном направлении (при его наличии)
            строго один сегмент (перелет без пересадок).

        Если выключен,
            то не накладывает дополнительную фильтрацию на возврат ссылок на прямое бронирование.
        """
        return self._is_flag_enabled('BOY_AFL_ONLY_DIRECT', ab_flags)

    def boy_afl_only_domestic(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Флаг ограничивающий поток только рейсами внутри России.

        Если включен совместно с BOY_AFL_ENABLE,
            то возвращает ссылку на прямое бронирование в том случае,
            если в вариантах перелета пункты отправления, прибытия и города/станции пересадок
            находятся внутри России.

        Если выключен,
            то не накладывает дополнительную фильтрацию на возврат ссылок на прямое бронирование.
        """
        return self._is_flag_enabled('BOY_AFL_ONLY_DOMESTIC', ab_flags)

    def boy_afl_not_business(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-20407: Флаг убирающий из потока бизнес-класс.

        Если включен совместно с BOY_AFL_ENABLE,
            то возвращает ссылку на прямое бронирование в том случае,
            если указан не бизнес-класс в запросе.

        Если выключен,
            то не накладывает дополнительную фильтрацию на возврат ссылок на прямое бронирование.
        """
        return self._is_flag_enabled('BOY_AFL_NOT_BUSINESS', ab_flags)

    def boy_afl_adults_only(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Редирект на собственное бронирование только вариантов,
            где поиск был совершен только на одного или несколько взрослых пассажиров.
            Исключаются дети.
        """
        return self._is_flag_enabled('BOY_AFL_ADULTS_ONLY', ab_flags)

    def boy_afl_only_beta_users(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-15476: Редирект на собственное бронирование только для бета пользователей."""
        return self._is_flag_enabled('BOY_AFL_ONLY_BETA_USERS', ab_flags)

    def boy_touch_afl_enable(self, ab_flags):
        """RASPTICKETS-16238: Редирект на собственное бронирование для тачей."""
        return self._is_flag_enabled('BOY_TOUCH_AFL_ENABLE', ab_flags)

    def boy_desktop_afl_enable(self, ab_flags):
        """RASPTICKETS-16238: Редирект на собственное бронирование для десктопов."""
        return self._is_flag_enabled('BOY_DESKTOP_AFL_ENABLE', ab_flags)

    def boy_afl_codeshares_with_pobeda(self, ab_flags):
        """RASPTICKETS-21479: Редиректим на BoY, если аэрофлот - кодшер победы"""
        return self._is_flag_enabled('BOY_AFL_ENABLE_CODESHARE_WITH_POBEDA', ab_flags)

    def boy_travel_app_afl_enable(self, ab_flags):
        # type: (dict) -> bool
        """RASPTICKETS-23109: Заблокировать переход на BoY для нового Аппа"""
        return self._is_flag_enabled('BOY_TRAVEL_APP_AFL_ENABLE', ab_flags)

    def _is_flag_enabled(self, flag, ab_flags):
        # type: (basestring, dict) -> bool
        """
            Приоритет у значений флагов экспериментов
        """
        if ab_flags.get(flag) == '0':
            return False
        return ab_flags.get(flag) == '1' or self._is_feature_flag_enabled(flag)

    def _is_feature_flag_enabled(self, key):
        # type: (basestring) -> bool
        return feature_flags_storage.flag_enabled(key)


afl_feature_flags = AeroflotBOYFlags()
