# coding=utf-8
from __future__ import unicode_literals
from urlparse import urljoin

import requests
import ujson as json
from django.conf import settings
from requests import ConnectionError, Timeout
from typing import Optional

from travel.library.python.tvm_ticket_provider import provider_fabric

from travel.avia.library.python.flask_helpers import safe_get_request_environ_value
from travel.avia.ticket_daemon_api.jsonrpc.lib.utils import skip_None_values


class InternalDaemonException(Exception):
    def __init__(self, message, code, response):
        self.message = message
        self.description = message
        self.code = code
        self.response = response

    def __str__(self):
        return "Internal Daemon Error [{}] [{}] [{}]".format(
            self.description,
            self.code,
            self.response
        )


class RedirectData(object):
    def __init__(self, url, post=None, m_url=None, query_source=None, marker=None):
        # type: (str, Optional[dict], Optional[str], Optional[str], Optional[str]) -> None
        self.url = url
        self.post = post
        self.m_url = m_url
        self.query_source = query_source
        self._marker = marker

    def get_marker(self, default=None):
        # type: (Optional[str]) -> str
        return self._marker if self._marker else default

    def __json__(self):
        return skip_None_values({
            'url': self.url,
            'post': self.post,
            'marker': self._marker,
        })

    def redirect_url(self, is_mobile=False):
        # type: (bool) -> str
        if is_mobile:
            return self.m_url or self.url
        else:
            return self.url

    def __str__(self):
        return "<RedirectData url={} post={} m_url={} query_source={}>, marker={}".format(
            self.url,
            self.post,
            self.m_url,
            self.query_source,
            self._marker,
        )


class InternalDaemonClient(object):
    BOY_VARIANT_TEST_CONTEXT_PARAM = 'variantTestContext'

    def __init__(self, tvm_provider, ticket_daemon_host=settings.TICKET_DAEMON_URL):
        self._tvm_provider = tvm_provider
        self._ticket_daemon_host = ticket_daemon_host

    def book(self, order_data, variant, flights, user_info,
             additional_data, utm_source, query_source, timeout, variant_test_context):
        data = self._request(
            method='post',
            route='book_redirect/',
            data={
                'order_data': order_data,
                'variant': variant,
                'flights': flights,
                'user_info': user_info,
                'utm_source': utm_source,
                'query_source': query_source,
                self.BOY_VARIANT_TEST_CONTEXT_PARAM: variant_test_context,
                'additional_data': additional_data,
            },
            timeout=timeout,
        ).json()

        return RedirectData(
            data['url'],
            data.get('post_data'),
            data.get('m_url'),
            query_source,
            data.get('marker'),
        )

    def redirect(self, order_data, user_info,
                 utm_source, query_source, additional_data,
                 timeout):
        back_flags = safe_get_request_environ_value('ab_experiment_raw_flags', default='')
        data = self._request(
            method='post',
            route='cook_redirect/',
            data={
                'order_data': order_data,
                'user_info': user_info,
                'utm_source': utm_source,
                'query_source': query_source,
                'additional_data': additional_data,
            },
            params={'back_flags': back_flags} if back_flags else None,
            timeout=timeout
        ).json()

        return RedirectData(
            data['url'],
            data.get('post_data'),
            data.get('m_url'),
            query_source,
            data.get('marker'),
        )

    def ping(self):
        self._request(
            method='get',
            route='ping',
            timeout=settings.DAEMON_API_HTTP_TIMEOUT
        )

    def query(self, query, ignore_cache, test_id, base_qid):
        params = {
            'qid': query.id,
            'ignore_cache': ignore_cache,
            'test_id': test_id,
            'partner_codes': query.partner_codes,
            'custom_store_time': query.meta.get('custom_store_time'),
            'base_qid': base_qid,
        }
        if query.meta.get('wizard_caches') is not None:
            params['wizard_caches'] = query.meta.get('wizard_caches')
        return self._request(
            method='get',
            route='query_partners_pure/',
            params=params,
            timeout=settings.DAEMON_API_HTTP_TIMEOUT,

        ).json()['data']
        # todo лучше возвращать не dict, а некий объект

    def start_track(self, data):
        return self._request(
            method='post',
            route='track/start/',
            data=data,
            timeout=settings.DAEMON_API_HTTP_TIMEOUT,
        ).content

    def track_result(self, trid):
        return self._request(
            method='get',
            route='track/result/',
            params={'trid': trid},
            timeout=settings.DAEMON_API_HTTP_TIMEOUT,
        ).content

    def _request(self, route, method, params=None, data=None, timeout=20):
        url = reduce(urljoin, [self._ticket_daemon_host, '/api/1.0/', route])

        headers = {
            'Content-type': 'application/json',
        }
        if settings.ENABLE_TVM:
            headers['X-YA-SERVICE-TICKET'] = self._tvm_provider.get_ticket(
                'AVIA_TICKET_HTTP_API_DAEMON_ID'
            )

        try:
            r = requests.request(
                method=method,
                url=url,
                data=json.dumps(data) if data else None,
                params=params,
                headers=headers,
                timeout=timeout,
                verify=False
            )
        except (ConnectionError, Timeout) as e:
            raise InternalDaemonException(
                u'Network problem: [%s] %r. Query: %r' % (url, e, data),
                499,
                None
            )

        if not r.ok:
            raise InternalDaemonException(
                u'Api error: %s [%s] %r. Query: %r' %
                (r.status_code, r.url, r.content, data),
                r.status_code,
                r.content
            )

        return r


internal_daemon_client = InternalDaemonClient(
    tvm_provider=provider_fabric.create(
        settings,
        timeout=settings.TVM_TIMEOUT,
    ),
    ticket_daemon_host=settings.TICKET_DAEMON_URL,
)
