# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from typing import Callable

from requests import ConnectionError, Timeout, HTTPError

from travel.avia.library.python.ticket_daemon.caches.operating_flights.cache import OperatingFlightProvider

logger = logging.getLogger(__name__)


class OperatingFlightCheckException(Exception):
    pass


class OperatingFlightChecker(object):
    def __init__(self, operating_flight_provider, airline_id, heuristics):
        # type: (OperatingFlightProvider, int, Callable[[basestring],bool])->None
        self._operating_flight_provider = operating_flight_provider
        self._airline_id = airline_id
        self._heuristics = heuristics

    def is_operating(self, segment):
        # type: (dict) -> bool
        try:
            return self._is_operating(segment)
        except Exception as e:
            raise OperatingFlightCheckException('Unexpected error', e)

    def _is_operating(self, segment):
        # type: (dict) -> bool
        segment_flight_number = segment['number']  # format: "SU 1234"
        operating_flight_params = self._operating_flight_provider.map_segment_to_params(segment)
        try:
            operating_flight = self._operating_flight_provider.get_operating_flight(*operating_flight_params)
        except (ConnectionError, Timeout, HTTPError):
            logger.exception('%s: applied heuristics', segment_flight_number)
            return self._heuristics(segment_flight_number)

        if operating_flight is None:
            return self._heuristics(segment_flight_number)

        return operating_flight.airlineID == self._airline_id
