from logging import getLogger, Logger

import travel.avia.ticket_daemon_api.jsonrpc.lib.partners_utils

from travel.avia.library.python.common.models.partner import DohopVendor


class RedirectTimeoutProvider(Exception):
    pass


class DisabledPartnerError(RedirectTimeoutProvider):
    def __init__(self, partner):
        self.partner = partner

    def __str__(self):
        return "Partner ({}, {}) is disabled ".format(
            self.partner.id,
            self.partner.code,
        )


class UnknownPartnerError(RedirectTimeoutProvider):
    def __init__(self, partner_code):
        self.partner_code = partner_code

    def __str__(self):
        return "Unknown partner {}".format(
            self.partner_code
        )


class PartnerRedirectTimeoutProvider(object):
    def __init__(self, logger, partners_utils, default_timeout,
                 dohop_timeout, custom_timeout):
        # type: (Logger, any) -> None
        self._logger = logger
        self._partners_utils = partners_utils
        self._default_timeout = default_timeout
        self._dohop_timeout = dohop_timeout
        self._custom_timeout_by_code = custom_timeout

    def _get_redirect_partner(self, partner_code):
        partner = self._partners_utils.get_partner_by_code(partner_code)

        if not partner:
            self._logger.warn('Unknown partner %s', partner)
            raise UnknownPartnerError(partner_code)

        if not partner.enabled:
            self._logger.warn('Disabled partner %s', partner)
            raise DisabledPartnerError(partner)

        return partner

    def get(self, partner_code):
        # type: (str) -> int
        partner = self._get_redirect_partner(partner_code)
        if isinstance(partner, DohopVendor):
            return self._dohop_timeout

        return self._custom_timeout_by_code.get(
            partner.code,
            self._default_timeout
        )


partner_redirect_timeout_provider = PartnerRedirectTimeoutProvider(
    logger=getLogger(__name__),
    partners_utils=travel.avia.ticket_daemon_api.jsonrpc.lib.partners_utils,
    default_timeout=5,
    dohop_timeout=15,
    custom_timeout={
        'pilotua': 10,
    }
)
