# -*- coding: utf-8 -*-
import hashlib
import logging
import six
from abc import abstractproperty, ABCMeta, abstractmethod
from datetime import datetime, timedelta
from itertools import chain, ifilter
from operator import itemgetter

from django.conf import settings
from pytz import timezone

from travel.avia.library.python.common.models.geo import Station
from travel.avia.ticket_daemon_api.jsonrpc.lib import feature_flags
from travel.avia.ticket_daemon_api.jsonrpc.lib.baggage import with_baggage
from travel.avia.ticket_daemon_api.jsonrpc.lib.date import get_msk_now, unixtime, get_utc_now
from travel.avia.ticket_daemon_api.jsonrpc.lib.fare_families.fare_families import FareFamilies
from travel.avia.ticket_daemon_api.jsonrpc.lib.result import Statuses

log = logging.getLogger(__name__)
BEFORE_DEPARTURE_MINUTES = 45
"""Допустимое время до вылета"""
AEROFLOT_BEFORE_DEPARTURE_MINUTES = 120

MAX_EASTERN_TIMEZONE = 14


def fill_tags(result_fares):
    """

    :type result_fares: dict
    """
    for tag, v in result_fares.iteritems():
        v['tag'] = tag

    return result_fares


def fill_fare_family(result_fares, result_flights, query):
    """Определяем fareGroup по описанию варианта и коду тарифа

    Пример формата варианта:
    {u'baggage': [[u'1p1p23d'], []],
     u'charter': None,
     u'created': 1574929832,
     u'expire': 1574931032,
     u'fare_codes': [[u'OSTD00OW'], []],
     u'route': [[u'1911292000UT8051911300150'], []],
     'tag': u'9f8e43b71538aede5d39853c626549a0',
     u'tariff': {u'currency': u'RUR', u'value': 15288.0}}
    """
    fare_families = FareFamilies()
    for tag, v in result_fares.iteritems():
        v['fare_families'] = [[None] * len(v['route'][0]), [None] * len(v['route'][1])]
        hash_value = hashlib.md5()
        for direction_idx, flights in enumerate(v['route']):
            for flight_idx, flight_key in enumerate(flights):
                code_tariff = v['fare_codes'][direction_idx][flight_idx]
                fare_family_tariff = fare_families.get_tariff(
                    code_tariff, result_flights[flight_key], query
                )
                v['fare_families'][direction_idx][flight_idx] = fare_family_tariff
                if fare_family_tariff:
                    hash_value.update(fare_family_tariff['key'].encode('utf-8'))
        if v.get('baggage'):
            hash_value.update(str(with_baggage(v.get('baggage'))))
        v['fare_families_hash'] = hash_value.hexdigest()


def validate_fare_families_for_variant(v, variant_ff):
    """Проверяем, что мы прочитали из кеша как раз такую структуру, какую ждём
    """
    route = v['route']
    if len(route) != len(variant_ff):
        log.error('invalid number of directions in fare families: route=%s, ff=%s', route, variant_ff)
        return False
    for variant_flights, route_flights in zip(variant_ff, route):
        if len(variant_flights) != len(route_flights):
            log.error('invalid number of flights in fare families: route=%s, ff=%s', route, variant_ff)
            return False
    return True


def fill_fare_family_from_variants_cache(fare_families, result_fares):
    """
    Добавляем в вариант fare_families, ранее сохранённые туда тикет-демоном.
    Пример fare_families можно посмотреть в тесте TestFareFamiliesFromCache.
    """
    ff_variants_map = {}
    ff_reference_map = {}

    fare_families_data = fare_families.get('data') if fare_families else None
    if not fare_families_data:
        fare_families_data = []

    # fare_families_data is expected to have zero or 1 element
    if fare_families_data:
        fare_families_data_elem = fare_families_data[0]
        variants_map = fare_families_data_elem.get('variantsMap')
        reference_map = fare_families_data_elem.get('fareFamilies')
        if variants_map and reference_map:
            ff_variants_map = variants_map
            ff_reference_map = reference_map

    for tag, v in six.iteritems(result_fares):
        variant_ff_data = ff_variants_map.get(tag, {})
        variant_ff = variant_ff_data.get('fare_families')
        found_ff_for_variant = variant_ff and validate_fare_families_for_variant(v, variant_ff)

        hash_value = hashlib.md5()
        v['fare_families'] = [[None] * len(v['route'][0]), [None] * len(v['route'][1])]
        for direction_idx, flights in enumerate(v['route']):
            for flight_idx, flight_key in enumerate(flights):
                fare_family_tariff = None
                flight_fare_family_key = variant_ff[direction_idx][flight_idx] if found_ff_for_variant else None
                if flight_fare_family_key:
                    fare_family_tariff = ff_reference_map.get(flight_fare_family_key)
                    if not fare_family_tariff:
                        log.error(
                            'inconsistent fare families map: missing key=%s, existing keys=%s, route=%s',
                            flight_fare_family_key,
                            ff_reference_map.keys(),
                            v['route'],
                        )
                        continue
                    v['fare_families'][direction_idx][flight_idx] = fare_family_tariff
                    hash_value.update(fare_family_tariff['key'].encode('utf-8'))

        if v.get('baggage'):
            hash_value.update(str(with_baggage(v.get('baggage'))))
        v['fare_families_hash'] = hash_value.hexdigest()


class CollectingModes(object):
    instant_search = 'instant_search'
    all = 'all'
    actual = 'actual'


class VariantsFabric(object):
    @staticmethod
    def create(
        unpacked_variants_data, status, query, partner_code,
        last_revision=0,
        mode=CollectingModes.instant_search,
        max_age=None,
    ):
        """
        :param unpacked_variants_data: распакованный словарь с данными вариантов из ydb
        :param basestring status: мод из Statuses
        :type query: travel.avia.ticket_daemon_api.jsonrpc.query.Query
        :type partner_code: basestring
        :param basestring mode: мод из CollectingModes
        :param int last_revision:
        :param max_age:
        :rtype: AbstractApiVariants
        """
        fares = unpacked_variants_data['fares']
        flights = unpacked_variants_data['flights']
        qid = unpacked_variants_data.get('qid')
        fare_families_data = unpacked_variants_data.get('fare_families_data')

        fill_tags(fares)  # fill_tags лучше вызывать в конце, после фильтров
        if feature_flags.fill_fare_family_enabled():
            if feature_flags.use_fare_families_from_variants_cache():
                fill_fare_family_from_variants_cache(fare_families_data, fares)
            else:
                fill_fare_family(fares, flights, query)

        latest_revision = max(last_revision, unpacked_variants_data['created'], *map(itemgetter('created'), fares.itervalues()))

        fares = fares.itervalues()

        if (
            feature_flags.replace_search_to_station_with_search_to_city()
            and query.base_qid is not None and query.base_qid != query.id
        ):
            fares = SubQueryFilter.filter(fares, flights, query)

        if max_age:
            fares = ByAge.filter(fares, timedelta(hours=max_age))

        if mode == CollectingModes.all:
            return ApiVariants(qid, flights, list(fares), status, unpacked_variants_data['query_time'], latest_revision)

        elif mode == CollectingModes.instant_search:
            is_outdated = unpacked_variants_data['expire'] < unixtime() and not last_revision
            status = Statuses.OUTDATED if (is_outdated and not last_revision) else status
            fares = InstantSearchFilter.filter(fares, unpacked_variants_data['expire'], last_revision)
            fares = ByCreatedFilter.filter(fares, latest_revision)
            fares = VariantsFabric.apply_departured_variants_filter(partner_code, fares, flights, query)
            return ApiVariants(qid, flights, list(fares), status, unpacked_variants_data['query_time'], latest_revision)

        elif mode == CollectingModes.actual:
            if unpacked_variants_data['expire'] < unixtime():
                fares = []
                flights = {}
            else:
                fares = ByCreatedFilter.filter(fares, unpacked_variants_data['created'])
                fares = VariantsFabric.apply_departured_variants_filter(partner_code, fares, flights, query)
            return ApiVariants(qid, flights, list(fares), status, unpacked_variants_data['query_time'], latest_revision)

    @staticmethod
    def apply_departured_variants_filter(partner_code, fares, flights, query):
        if partner_code == settings.AFL_PARTNER_CODE:
            fares = DeparturedVariantsFilter(AEROFLOT_BEFORE_DEPARTURE_MINUTES).filter(fares, flights, query)
        else:
            fares = DeparturedVariantsFilter(BEFORE_DEPARTURE_MINUTES).filter(fares, flights, query)
        return fares


class AbstractSimpleApiVariants(object):
    __metaclass__ = ABCMeta
    variants = abstractproperty()
    flights = abstractproperty()
    query_time = abstractproperty()

    @abstractmethod
    def __len__(self):
        pass

    @abstractmethod
    def __nonzero__(self):
        pass


class AbstractApiVariants(AbstractSimpleApiVariants):
    revision = abstractproperty()
    status = abstractproperty()
    qid = abstractproperty()

    @abstractmethod
    def filter(self, function):
        pass


class SimpleApiVariants(AbstractSimpleApiVariants):
    def __init__(self, flights, fares, query_time=0, **kwargs):
        self._flights = flights
        self._variants = fares
        self._query_time = query_time

    @property
    def variants(self):
        return self._variants

    @property
    def flights(self):
        return self._flights

    @property
    def query_time(self):
        return self._query_time

    def __len__(self):
        return len(self._variants)

    def __nonzero__(self):
        return bool(len(self))


class ApiVariants(SimpleApiVariants, AbstractApiVariants):
    def __init__(self, qid, flights, fares, status, query_time, revision):
        """

        :param dict flights:
        :param list fares:
        :param basestring status:
        :param int query_time:
        :param int revision: unix timestamp самого свежего из добавленных вариантов # todo: пока работа с ними не очень удобна
        """

        SimpleApiVariants.__init__(self, flights, fares, query_time)
        self._qid = qid
        self._status = status
        self._revision = revision

    @property
    def qid(self):
        return self._qid

    @property
    def flights(self):
        return self._flights_for_variants(self._variants, self._flights)

    @property
    def status(self):
        return self._status

    @property
    def revision(self):
        return self._revision

    def filter(self, function):
        self._variants = filter(function, self._variants)

    def _flights_for_variants(self, variants, flights):
        flight_keys = {f_k for v in variants for f_k in chain.from_iterable(v['route'])}
        return {k: v for k, v in flights.iteritems() if k in flight_keys}


class ByCreatedFilter(object):
    @staticmethod
    def filter(variants, created_at):
        """
        Оставляем варианты, созданные в момент created_at или раньше не более чем на 120 секунд
        :param int created_at: unix timestamp, крайняя правая граница времени создания

        """
        left_border = created_at - settings.PARTNER_QUERY_TIMEOUT

        return ifilter(
            lambda v: left_border <= v['created'] <= created_at,
            variants
        )


class ByAge(object):
    @staticmethod
    def filter(variants, max_age_timedelta):
        """

        :param variants:
        :param max_age_timedelta:
        :rtype: Iterable[Dict]
        """

        max_allowed_created_timestamp = unixtime() - max_age_timedelta.total_seconds()

        return ifilter(
            lambda v: max_allowed_created_timestamp <= v['created'],
            variants
        )


class DeparturedVariantsFilter(object):
    def __init__(self, before_departure_minutes):
        self._before_departure_minutes = before_departure_minutes

    def filter(self, variants, flights, query):
        """

        :param variants:
        :param flights:
        :param query:
        :rtype: Iterable[Dict]
        """
        try:
            if query.date_forward != query.point_from.local_time.date():
                return variants
        except ValueError:
            pass

        if query.date_forward > (get_utc_now() + timedelta(hours=MAX_EASTERN_TIMEZONE)).date():
            return variants

        actual_routes = self._get_actual_routes(flights)

        return ifilter(
            lambda v: all(r in actual_routes for r in chain.from_iterable(v['route'])),
            variants
        )

    def _get_actual_routes(self, flights):
        """
        departure": {
            "local": "2017-09-01T03:00:00",
            "tzname": "Europe/Moscow",
            "offset": 180.0,
        }
        """
        actual_routes = set()
        now = get_msk_now()
        min_departure = now + timedelta(minutes=self._before_departure_minutes)

        for route, flight in flights.iteritems():
            # todo: такая штука по работе с датой, наверное, уже написана раз 5
            departure_unaware_dt = datetime.strptime(
                flight['departure']['local'], '%Y-%m-%dT%H:%M:%S'
            )
            local_tz = timezone(flight['departure']['tzname'])
            departure_aware_dt = local_tz.localize(departure_unaware_dt)

            if min_departure < departure_aware_dt:
                actual_routes.add(route)
        return actual_routes


class InstantSearchFilter(object):
    @staticmethod
    def filter(variants, result_expired, last_revision):
        """
         если вариант старый, то отправляем все,
         если пришли новые результаты,
         то фильтруем свежие и те, которые ещё не отправляли
        :param variants:
        :param result_expired:
        :param last_revision:
        :rtype: Iterator
        """
        now = unixtime()
        is_outdated = result_expired < now and not last_revision

        return ifilter(
            lambda v: (not last_revision or last_revision < v['created']) and (is_outdated or now < v['expire']),
            variants
        )


class SubQueryFilter(object):
    @staticmethod
    def filter(variants, flights, query):
        """
         Фильтруем варианты от/до станций при поисках среди вариантов до городов этих станций
        :param variants:
        :param flights:
        :type query: jsonrpc.query.Query
        :rtype: Iterator
        """
        station_from_id, station_to_id = None, None

        if query.base_query.point_from.type == Station:
            station_from_id = int(query.base_query.point_from.id)
        if query.base_query.point_to.type == Station:
            station_to_id = int(query.base_query.point_to.id)

        for variant in variants:
            if station_from_id is not None:
                first_segment_forward_route_key = variant['route'][0][0]
                if flights[first_segment_forward_route_key]['from'] != station_from_id:
                    continue
                if len(variant['route'][1]):
                    last_segment_backward_route_key = variant['route'][1][-1]
                    if flights[last_segment_backward_route_key]['to'] != station_from_id:
                        continue
            if station_to_id is not None:
                last_segment_forward_route_key = variant['route'][0][-1]
                if flights[last_segment_forward_route_key]['to'] != station_to_id:
                    continue
                if len(variant['route'][1]):
                    first_segment_backward_route_key = variant['route'][1][0]
                    if flights[first_segment_backward_route_key]['from'] != station_to_id:
                        continue

            yield variant
