# -*- coding: utf-8 -*-
import logging
from typing import Iterator, Tuple

from travel.avia.ticket_daemon_api.jsonrpc.lib.date import unixtime
from travel.avia.ticket_daemon_api.jsonrpc.lib.result.collector.utils import safe_deserialize_packed
from travel.avia.ticket_daemon_api.jsonrpc.lib.ydb import cache as service_cache
from travel.avia.ticket_daemon_api.jsonrpc.query import Query


REDIRECT_DATA_CACHE_TIME = 6 * 60 * 60
log = logging.getLogger(__name__)


def get_variants_and_redirect_data(query, partner_codes, fetch_redirect_data=False):
    """Новый result_fetcher"""
    columns_to_select = ('partner_code', 'variants', 'created_at')
    if fetch_redirect_data:
        columns_to_select = ('partner_code', 'variants', 'created_at', 'redirect_data')

    for row in service_cache.ServiceCache.get_many(query, partner_codes, columns=columns_to_select):
        yield row['partner_code'], row


def get_variants(results):
    # type: (Tuple(str, dict)) -> Iterator[Tuple(str, dict)]
    """Лениво распаковывающий итератор для variants из ydb."""
    for partner_code, row in results:
        yield partner_code, safe_deserialize_packed(row['variants'], log)


def fetch_and_get_variants(query, partner_codes):
    # type: (Query, list[str]) -> Iterator[Tuple(str, dict)]
    for partner_code, variants in get_variants(get_variants_and_redirect_data(query, partner_codes)):
        yield partner_code, variants


def get_redirect_data(results):
    # type: (Tuple(str, dict)) -> Iterator[Tuple(str, dict)]
    """Лениво распаковывающий итератор для redirect_data из ydb."""
    for partner_code, row in results:
        unix_time = unixtime()
        if row.get('created_at') and unix_time - row['created_at'] >= REDIRECT_DATA_CACHE_TIME:
            continue
        yield partner_code, safe_deserialize_packed(row['redirect_data'], log)
