# -*- coding: utf-8 -*-
"""
ContDetails - пример данных:
{
  "variants_count": 200,
  "qid": "210924-090435-666.mavia-travel.plane.c65_c239_2021-10-24_2021-10-27_economy_1_0_0_ru.ru",
  "revisions": {
    "kiwi": 1632364764,
    "kupibilet": 1632364761,
    "megotravel": 1632364762,
    "aeroflot": 1632463483,
    "aviakass": 1632364756,
    "biletix": 1632364760,
    "biletdv": 1632364757,
    "citytravel1": 1632364758
  },
  "cont": 3,
  "skip_partner_codes": [
    "dohop_175",
    "amadeus_sq",
    "superkassa",
    "agent",
    "flyone",
    "ticketsru",
    "redwings",
    "charterbilet",
    "nebotravel",
    "trip_ru",
    "anywayanyday",
    "rusline",
    "utair",
    "azimuth",
  ]
}
"""
from collections import namedtuple

import typing
import ujson

from travel.avia.ticket_daemon_api.jsonrpc.lib.caches import default_cache

CONTINUATION_STORE_TIME = 60
ContDetails = namedtuple(
    'ContDetails', ['qid', 'cont', 'skip_partner_codes', 'revisions', 'variants_count']
)


def _cont_key(qid, cont):
    return '/{}/cont/{}'.format(qid, cont)


def _rasp_cont_key(qid):
    return '/{}/rasp_cont'.format(qid)


def get_cont_details(qid, cont):
    # type: (basestring, int) -> typing.Optional[ContDetails]
    if cont == 0:
        return ContDetails(qid, cont, set(), {}, 0)

    raw = default_cache.get(_cont_key(qid, cont))

    if not raw:
        return None

    d = ujson.loads(raw)
    d['skip_partner_codes'] = set(d['skip_partner_codes'])

    return ContDetails(**d)


def get_rasp_cont_details(qid):
    # type: (basestring, int) -> typing.Optional[ContDetails]

    raw = default_cache.get(_rasp_cont_key(qid))

    if not raw:
        return ContDetails(qid, 0, set(), {}, 0)

    d = ujson.loads(raw)
    d['skip_partner_codes'] = set(d['skip_partner_codes'])

    return ContDetails(**d)


def create_next_cont(cont_details, skip_partner_codes, revisions, in_progress, variants_count, is_rasp_cont=False):
    """

    :param ContDetail cont_details: Контекст предыдущего запроса
    :param typing.Set[string] skip_partner_codes: Список партнеров, по которым
     отданы все доступные варианты в текущем запросе и закончен опрос
    :param typing.Dict[string, int] revisions: created_at самого свежего
     варианта по отданным партнерам
    :param typing.List[string] in_progress: Список партнеров, по которым будут
     отданы результаты в следующих запросах или ещё не окончен опрос
    :param int variants_count: Количество актуальных вариантов переданных на фронт
    :param bool is_rasp_cont: Сохранять ContDetail для распов
    :return:
    """
    # Закончили опрашивать
    if not in_progress:
        return ContDetails(
            cont_details.qid, cont=None,
            skip_partner_codes=set(skip_partner_codes),
            revisions=revisions,
            variants_count=variants_count,
        )

    if not skip_partner_codes and revisions == cont_details.revisions:
        return cont_details

    next_skip_partner_codes = cont_details.skip_partner_codes | set(skip_partner_codes)
    next_cont = cont_details.cont + 1
    next_variants_count = cont_details.variants_count + variants_count

    next_cont_details = ContDetails(cont_details.qid, next_cont, next_skip_partner_codes, revisions,
                                    next_variants_count)
    default_cache.set(
        key=_rasp_cont_key(cont_details.qid) if is_rasp_cont else _cont_key(cont_details.qid, next_cont),
        value=ujson.dumps(next_cont_details._asdict()),
        timeout=CONTINUATION_STORE_TIME,
    )
    return next_cont_details
