# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import datetime, timedelta

import gevent
import pytz
from django.conf import settings
from statsd.defaults.django import statsd
from retrying import retry

from travel.avia.library.python.common.saas.index import LoggingIndex, SaasIndex, ISaasIndex
from travel.avia.library.python.common.saas.search import LoggingSearch, SaasSearch, ISaasSearch
from travel.avia.library.python.common.utils import environment
from travel.avia.library.python.common.utils.iterrecipes import group_by

from travel.avia.ticket_daemon_api.jsonrpc.lib.yt_loggers.yt_logger import YtLogger

log = logging.getLogger(__name__)
yt_logger = YtLogger('yt.saas_index', environment=environment)


class SaasException(Exception):
    def __init__(self, message):
        super(SaasException, self).__init__(message)
        self.description = 'Saas error. Exception: %s' % message

    def __str__(self):
        return self.description


def patch_response(function_, key, type_):
    prefix = 'saas.%s.%s' % (key, type_)

    def wrapped_function(*args, **kwargs):
        response = function_(*args, **kwargs)

        statsd.timing('%s.elapsed' % prefix, response.elapsed.total_seconds() * 1000)
        statsd.incr('%s.status_code.%d' % (prefix, response.status_code))
        return response
    return wrapped_function


class ResponseLoggingSaas(object):
    def __init__(self, index, search, key):
        # type: (ISaasIndex, ISaasSearch) -> None
        self._index = index
        self._index._origin._session.post = patch_response(
            self._index._origin._session.post, key, 'index'
        )
        self._search = search
        self._search._origin._session.get = patch_response(
            self._index._origin._session.get, key, 'search'
        )

    def index(self, key, *args, **kwargs):
        return self._index.index(key, *args, **kwargs)

    def delete(self, key, *args, **kwargs):
        return self._index.delete(key, *args, **kwargs)

    @retry(stop_max_attempt_number=3, wait_fixed=200)
    def search(self, keys, *args, **kwargs):
        return self._search.search(keys, *args, **kwargs)


def _saas_key(key):
    return '{}/{}'.format(settings.SAAS_CACHE_PREFIX, key)


SAAS_BY_SERVICE = {}
for saas_service, config in settings.SAAS_SERVICES.iteritems():
    SAAS_BY_SERVICE[saas_service] = ResponseLoggingSaas(
        index=LoggingIndex(SaasIndex(
            index_host=settings.SAAS_INDEX_HOST,
            service_key=config.key,
            retry_5xx=True,
        )),
        search=LoggingSearch(SaasSearch(
            search_host=settings.SAAS_SEARCH_HOST,
            service_name=config.name,
            retry_5xx=True,
        )),
        key=saas_service,
        )


def get_saas_client(key):
    if 'redirect_data' in key:
        return SAAS_BY_SERVICE['redirect_data'], 'redirect'
    elif 'big_beauty' in key:
        return SAAS_BY_SERVICE['big_beauty'], 'big_beauty'
    else:
        return SAAS_BY_SERVICE['result'], 'flights'


class BigBeautyCache(object):
    SAAS_SERVICE = SAAS_BY_SERVICE['big_beauty']

    DEFAULT_COLUMN_WITH_VARIANTS = 'protobuf'
    FILTER_STATE_COLUMN = 'filter_state'

    AVAILABLE_SORTS = ('protobuf', 'sorted_by_price', 'control_with_weekdays', 'front_sort', 'kateov_sort')

    @classmethod
    def set(cls, key, variants, filter_state, store_time, options=None):
        doc = {
            cls.DEFAULT_COLUMN_WITH_VARIANTS: variants,
            cls.FILTER_STATE_COLUMN: filter_state,
        }
        _set(cls.SAAS_SERVICE, key, doc, expires_at(store_time), options)

    @classmethod
    def get(cls, key, columns=(DEFAULT_COLUMN_WITH_VARIANTS, FILTER_STATE_COLUMN)):
        new_key = _saas_key(key)
        try:
            result = cls.SAAS_SERVICE.search([new_key], columns=columns)
            if result:
                return result[new_key]
        except Exception as exc:
            log.exception('Unexpected SaaS search error: %r', exc)
            raise SaasException(exc)
        return None


class SaasResultsCache(object):
    SAAS_SERVICE = SAAS_BY_SERVICE['result']
    __columns = ('flights', 'meta')
    COLUMNS = namedtuple('COLUMNS', __columns)(*__columns)

    @classmethod
    def set(cls, key, search_result, meta, store_time, options=None):
        doc = {
            cls.COLUMNS.flights: search_result,
            cls.COLUMNS.meta: meta,
        }

        _set(cls.SAAS_SERVICE, key, doc, expires_at(store_time), options)

    @classmethod
    def get(cls, key, columns=COLUMNS):
        new_key = _saas_key(key)
        try:
            result = cls.SAAS_SERVICE.search([new_key], columns=columns)
            if result:
                return result[new_key]
        except Exception as exc:
            log.exception('Unexpected SaaS search error: %r', exc)
        return None

    @classmethod
    def get_many(cls, keys, columns=COLUMNS):
        new_keys = {_saas_key(key): key for key in keys}
        try:
            result = cls.SAAS_SERVICE.search(new_keys.keys(), columns=columns)
            if result:
                return {new_keys[key]: v for key, v in result.iteritems()}
        except Exception as exc:
            log.exception('Unexpected SaaS search error: %r', exc)
        return None


def _set(saas_service, key, doc, expires_at, options=None):
    try:
        saas_service.index(
            key=_saas_key(key),
            doc=doc,
            expires_at=expires_at,
            options=options,
        )
        yt_logger.log({'key': key, 'expires_at': expires_at})
    except Exception as exc:
        log.exception('Unexpected SaaS index error: %r', exc)


def set(key, value, store_time, options=None):
    """

    :type key: unicode
    :type value: str
    :param int store_time: in seconds
    :type options: dict
    """
    _expires_at = expires_at(store_time)
    _saas, column = get_saas_client(key)
    _set(_saas, key, {column: value}, _expires_at, options)


def set_many(data, store_time, options=None):
    """

    :param dict data: Dict[unicode, str]
    :param int store_time: in seconds
    :type options: dict
    """
    gevent.joinall([
        gevent.spawn(
            set, key, value,
            expires_at=expires_at(store_time), options=options
        )
        for key, value in data.iteritems()
    ])


def get_many(keys):
    new_to_old = {_saas_key(key): key for key in keys}

    result = {}
    try:
        for (_saas, column), keys in group_by(new_to_old.keys(), get_saas_client):
            for key, doc in _saas.search(keys, columns=(column, )).iteritems():
                result[new_to_old[key]] = doc[column]

    except Exception as exc:
        log.exception('Unexpected SaaS search error: %r', exc)
        raise SaasException(exc)
    return result


def get(key):
    result = get_many([key])
    return result.get(key)


def expires_at(store_time):
    return datetime.now(pytz.UTC) + timedelta(seconds=store_time)
