# coding: utf-8
import hashlib
from typing import Optional

from django.conf import settings


class TariffSerializer(object):
    _secret = settings.SECRET_KEY

    @classmethod
    def serialize(cls, tariff, created):
        # type: (dict) -> str
        """Подпись тарифа для обратной передачи от фронта в td-api"""
        h = cls._hash(tariff.get('value'), tariff.get('currency'))
        parts = map(str, [tariff.get('value'), tariff.get('currency'), h])
        if created is not None:
            parts.append(str(created))
        return '|'.join(parts)

    @classmethod
    def deserialize(cls, tariff):
        # type: (str) -> Optional[dict]
        """Проверка подписи и формирование словаря тарифа"""
        data = tariff.split('|')
        price_unixtime = None
        if len(data) == 4:
            price_unixtime = data.pop()

        if len(data) != 3:
            return None

        if cls._hash(data[0], data[1]) != data[2]:
            return None

        return {
            'value': data[0],
            'currency': data[1],
            'price_unixtime': price_unixtime,
        }

    @classmethod
    def _hash(cls, price, currency):
        # type: (str, str) -> str
        m = hashlib.md5()
        m.update('{}|{}'.format(price, currency) + cls._secret)

        return m.hexdigest()
